import { useState } from 'react';
import { Link } from 'react-router-dom';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import {
  faMagnifyingGlass,
  faHeadset,
  faTicket,
  faBook,
  faUsers,
  faChartLine,
  faArrowRight,
  faEnvelope,
  faClock
} from '@fortawesome/free-solid-svg-icons';
import { faDiscord } from '@fortawesome/free-brands-svg-icons';
import Navbar from '../components/Navbar';
import Footer from '../components/Footer';

const Support = () => {
  const [searchQuery, setSearchQuery] = useState('');

  const handleSearch = (e) => {
    e.preventDefault();
    if (searchQuery.trim()) {
      window.location.href = `/community?q=${encodeURIComponent(searchQuery)}`;
    }
  };

  const supportChannels = [
    {
      icon: faTicket,
      title: 'Support Tickets',
      description: 'Create a ticket for technical issues or billing inquiries. Average response time: 15 minutes.',
      buttonText: 'Open Ticket',
      href: '/support/ticket',
      color: '#0f62fe'
    },
    {
      icon: faDiscord,
      title: 'Discord Community',
      description: 'Join our Discord server for community support, announcements, and direct help from staff.',
      buttonText: 'Join Discord',
      href: 'https://discord.gg/Xzk5QHKJuC',
      external: true,
      color: '#5865F2'
    },
    {
      icon: faEnvelope,
      title: 'Email Support',
      description: 'For non-urgent inquiries or business partnerships, email us directly.',
      buttonText: 'Send Email',
      href: 'mailto:support@hostcraft.site',
      external: true,
      color: '#24a148'
    }
  ];

  const quickLinks = [
    {
      icon: faBook,
      title: 'Documentation',
      description: 'Technical guides and API reference',
      href: '/docs'
    },
    {
      icon: faUsers,
      title: 'Community Guides',
      description: 'User-contributed tutorials',
      href: '/community'
    },
    {
      icon: faChartLine,
      title: 'Status Page',
      description: 'Real-time network status',
      href: '#'
    }
  ];

  return (
    <div className="pt-14 bg-[#f4f4f4] min-h-screen">
      {/* Hero Section */}
      <section className="bg-[#161616] py-20 border-b border-[#393939] relative overflow-hidden">
        {/* Technical Grid Pattern */}
        <div
          className="absolute inset-0 z-0 opacity-10 pointer-events-none"
          style={{
            backgroundImage: 'linear-gradient(#ffffff 1px, transparent 1px), linear-gradient(to right, #ffffff 1px, transparent 1px)',
            backgroundSize: '40px 40px'
          }}
        ></div>

        <div className="bx--grid relative z-10">
          <div className="max-w-3xl mx-auto text-center">
            <div className="inline-flex items-center gap-2 px-3 py-1 bg-[#0f62fe]/10 border border-[#0f62fe] text-[#0f62fe] text-xs font-bold uppercase tracking-wider mb-6 rounded-full">
              <FontAwesomeIcon icon={faHeadset} />
              Support Center
            </div>

            <h1 className="text-4xl md:text-5xl font-light text-white mb-6 tracking-tight">
              How Can We <span className="font-bold text-[#0f62fe]">Help?</span>
            </h1>

            <p className="text-lg text-[#a8a8a8] mb-8">
              Our engineering team is available 24/7 to assist with any technical or billing issues.
            </p>

            {/* Search Input */}
            <form onSubmit={handleSearch} className="relative max-w-xl mx-auto">
              <div className="absolute inset-y-0 left-0 pl-5 flex items-center pointer-events-none">
                <FontAwesomeIcon icon={faMagnifyingGlass} className="text-[#6f6f6f]" />
              </div>
              <input
                type="text"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                placeholder="Search for help..."
                className="w-full pl-12 pr-4 py-4 bg-[#262626] border border-[#393939] text-white rounded-lg focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] outline-none transition placeholder-[#6f6f6f]"
              />
            </form>

            {/* Response Time Badge */}
            <div className="inline-flex items-center gap-2 mt-6 px-4 py-2 bg-[#262626] rounded-full">
              <FontAwesomeIcon icon={faClock} className="text-[#24a148]" />
              <span className="text-sm text-[#a8a8a8]">
                Average response time: <span className="text-[#24a148] font-bold">15 minutes</span>
              </span>
            </div>
          </div>
        </div>
      </section>

      {/* Support Channels */}
      <section className="py-20">
        <div className="bx--grid">
          <div className="max-w-5xl mx-auto">
            <h2 className="text-2xl font-bold text-[#161616] mb-8 text-center">Contact Us</h2>

            <div className="grid md:grid-cols-3 gap-6">
              {supportChannels.map((channel, index) => (
                <div
                  key={index}
                  className="bg-white border border-[#e0e0e0] rounded-xl p-8 hover:shadow-lg transition text-center"
                >
                  <div
                    className="w-16 h-16 rounded-full flex items-center justify-center mx-auto mb-6"
                    style={{ backgroundColor: `${channel.color}10`, color: channel.color }}
                  >
                    <FontAwesomeIcon icon={channel.icon} className="text-2xl" />
                  </div>

                  <h3 className="text-xl font-bold text-[#161616] mb-3">{channel.title}</h3>
                  <p className="text-[#6f6f6f] text-sm mb-6">{channel.description}</p>

                  {channel.external ? (
                    <a
                      href={channel.href}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="inline-flex items-center gap-2 px-6 py-3 font-bold rounded transition text-white"
                      style={{ backgroundColor: channel.color }}
                    >
                      {channel.buttonText}
                      <FontAwesomeIcon icon={faArrowRight} />
                    </a>
                  ) : (
                    <Link
                      to={channel.href}
                      className="inline-flex items-center gap-2 px-6 py-3 font-bold rounded transition text-white"
                      style={{ backgroundColor: channel.color }}
                    >
                      {channel.buttonText}
                      <FontAwesomeIcon icon={faArrowRight} />
                    </Link>
                  )}
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Quick Links */}
      <section className="py-16 bg-white border-y border-[#e0e0e0]">
        <div className="bx--grid">
          <div className="max-w-4xl mx-auto">
            <h2 className="text-2xl font-bold text-[#161616] mb-8 text-center">Self-Service Resources</h2>

            <div className="grid md:grid-cols-3 gap-6">
              {quickLinks.map((link, index) => (
                <Link
                  key={index}
                  to={link.href}
                  className="flex items-start gap-4 p-6 border border-[#e0e0e0] rounded-lg hover:border-[#0f62fe] hover:shadow-md transition group"
                >
                  <div className="w-10 h-10 bg-[#f4f4f4] text-[#6f6f6f] rounded-lg flex items-center justify-center group-hover:bg-[#0f62fe]/10 group-hover:text-[#0f62fe] transition">
                    <FontAwesomeIcon icon={link.icon} />
                  </div>
                  <div>
                    <h3 className="font-bold text-[#161616] mb-1">{link.title}</h3>
                    <p className="text-sm text-[#6f6f6f]">{link.description}</p>
                  </div>
                </Link>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* FAQ Preview */}
      <section className="py-20">
        <div className="bx--grid">
          <div className="max-w-3xl mx-auto text-center">
            <h2 className="text-2xl font-bold text-[#161616] mb-4">Common Questions</h2>
            <p className="text-[#6f6f6f] mb-8">
              Check our documentation for answers to frequently asked questions.
            </p>

            <div className="space-y-4 text-left">
              {[
                { q: 'How do I connect to my VPS via RDP?', a: '/community' },
                { q: 'What payment methods do you accept?', a: '/community' },
                { q: 'How do I reset my root/admin password?', a: '/community/reset-root-password' },
                { q: 'Is DDoS protection included?', a: '/community/ddos-filters' }
              ].map((faq, index) => (
                <Link
                  key={index}
                  to={faq.a}
                  className="flex items-center justify-between p-4 bg-white border border-[#e0e0e0] rounded-lg hover:border-[#0f62fe] hover:shadow-md transition group"
                >
                  <span className="font-medium text-[#161616]">{faq.q}</span>
                  <FontAwesomeIcon
                    icon={faArrowRight}
                    className="text-[#a8a8a8] group-hover:text-[#0f62fe] group-hover:translate-x-1 transition-all"
                  />
                </Link>
              ))}
            </div>

            <Link
              to="/community"
              className="inline-flex items-center gap-2 mt-8 text-[#0f62fe] font-bold hover:underline"
            >
              View All Guides
              <FontAwesomeIcon icon={faArrowRight} />
            </Link>
          </div>
        </div>
      </section>
    </div>
  );
};

export default Support;
