import { useState } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faUserPlus, faSpinner } from '@fortawesome/free-solid-svg-icons';

const Register = () => {
  const [formData, setFormData] = useState({
    username: '',
    email: '',
    password: ''
  });
  const [loading, setLoading] = useState(false);
  const [alert, setAlert] = useState({ show: false, type: '', message: '' });
  const navigate = useNavigate();

  const handleChange = (e) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value
    });
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    setAlert({ show: false, type: '', message: '' });

    // Validate username
    if (formData.username.length < 3 || formData.username.length > 20) {
      setAlert({
        show: true,
        type: 'error',
        message: 'Username must be between 3 and 20 characters'
      });
      setLoading(false);
      return;
    }

    // Validate password
    if (formData.password.length < 6) {
      setAlert({
        show: true,
        type: 'error',
        message: 'Password must be at least 6 characters'
      });
      setLoading(false);
      return;
    }

    // Simulate API call
    setTimeout(() => {
      setAlert({
        show: true,
        type: 'success',
        message: 'Account created successfully! Redirecting to login...'
      });

      setTimeout(() => {
        navigate('/login');
      }, 2000);

      setLoading(false);
    }, 1500);
  };

  return (
    <div className="min-h-screen bg-[#161616] flex items-center justify-center px-4 py-20">
      {/* Background Pattern */}
      <div
        className="absolute inset-0 z-0 opacity-[0.03] pointer-events-none"
        style={{
          backgroundImage: 'linear-gradient(#fff 1px, transparent 1px), linear-gradient(to right, #fff 1px, transparent 1px)',
          backgroundSize: '40px 40px'
        }}
      ></div>

      <div className="w-full max-w-md relative z-10">
        {/* Logo */}
        <div className="text-center mb-8">
          <Link to="/" className="inline-block">
            <span className="font-bold text-2xl text-white">
              VCH<span className="font-light text-[#a8a8a8]">CLOUD</span>
            </span>
          </Link>
          <p className="text-[#a8a8a8] mt-2">Create your account</p>
        </div>

        {/* Form Card */}
        <div className="bg-[#262626] border border-[#393939] rounded-lg p-8">
          {/* Alert */}
          {alert.show && (
            <div
              className={`mb-6 p-4 rounded border ${
                alert.type === 'success'
                  ? 'bg-[#defbe6] border-[#24a148] text-[#0e6027]'
                  : 'bg-[#fff1f1] border-[#da1e28] text-[#750e13]'
              }`}
            >
              {alert.message}
            </div>
          )}

          <form onSubmit={handleSubmit} className="space-y-6">
            {/* Username Field */}
            <div>
              <label htmlFor="username" className="block text-sm font-medium text-[#c6c6c6] mb-2">
                Username
              </label>
              <input
                type="text"
                id="username"
                name="username"
                value={formData.username}
                onChange={handleChange}
                required
                minLength={3}
                maxLength={20}
                pattern="[a-zA-Z0-9]+"
                className="w-full px-4 py-3 bg-[#161616] border border-[#393939] text-white rounded focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] outline-none transition placeholder-[#6f6f6f]"
                placeholder="Choose a username"
              />
              <p className="text-xs text-[#6f6f6f] mt-1">3-20 characters, letters and numbers only</p>
            </div>

            {/* Email Field */}
            <div>
              <label htmlFor="email" className="block text-sm font-medium text-[#c6c6c6] mb-2">
                Email Address
              </label>
              <input
                type="email"
                id="email"
                name="email"
                value={formData.email}
                onChange={handleChange}
                required
                className="w-full px-4 py-3 bg-[#161616] border border-[#393939] text-white rounded focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] outline-none transition placeholder-[#6f6f6f]"
                placeholder="Enter your email"
              />
            </div>

            {/* Password Field */}
            <div>
              <label htmlFor="password" className="block text-sm font-medium text-[#c6c6c6] mb-2">
                Password
              </label>
              <input
                type="password"
                id="password"
                name="password"
                value={formData.password}
                onChange={handleChange}
                required
                minLength={6}
                className="w-full px-4 py-3 bg-[#161616] border border-[#393939] text-white rounded focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] outline-none transition placeholder-[#6f6f6f]"
                placeholder="Create a password"
              />
              <p className="text-xs text-[#6f6f6f] mt-1">Minimum 6 characters</p>
            </div>

            {/* Submit Button */}
            <button
              type="submit"
              disabled={loading}
              className="w-full py-4 bg-[#0f62fe] text-white font-bold rounded hover:bg-[#0353e9] transition flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {loading ? (
                <>
                  <FontAwesomeIcon icon={faSpinner} className="animate-spin" />
                  Processing...
                </>
              ) : (
                <>
                  <FontAwesomeIcon icon={faUserPlus} />
                  Create Account
                </>
              )}
            </button>
          </form>

          {/* Login Link */}
          <p className="text-center text-[#a8a8a8] mt-6">
            Already have an account?{' '}
            <Link to="/login" className="text-[#0f62fe] font-medium hover:underline">
              Sign in
            </Link>
          </p>
        </div>
      </div>
    </div>
  );
};

export default Register;
