import { useState } from 'react';
import { Link, useNavigate } from 'react-router-dom';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faArrowRight, faSpinner } from '@fortawesome/free-solid-svg-icons';

const Login = () => {
  const [formData, setFormData] = useState({
    login: '',
    password: ''
  });
  const [loading, setLoading] = useState(false);
  const [alert, setAlert] = useState({ show: false, type: '', message: '' });
  const navigate = useNavigate();

  const handleChange = (e) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value
    });
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    setAlert({ show: false, type: '', message: '' });

    // Simulate API call
    setTimeout(() => {
      // For demo purposes, show success
      setAlert({
        show: true,
        type: 'success',
        message: 'Login successful! Redirecting...'
      });

      setTimeout(() => {
        navigate('/dashboard');
      }, 1000);

      setLoading(false);
    }, 1500);
  };

  return (
    <div className="min-h-screen bg-[#161616] flex items-center justify-center px-4 py-20">
      {/* Background Pattern */}
      <div
        className="absolute inset-0 z-0 opacity-[0.03] pointer-events-none"
        style={{
          backgroundImage: 'linear-gradient(#fff 1px, transparent 1px), linear-gradient(to right, #fff 1px, transparent 1px)',
          backgroundSize: '40px 40px'
        }}
      ></div>

      <div className="w-full max-w-md relative z-10">
        {/* Logo */}
        <div className="text-center mb-8">
          <Link to="/" className="inline-block">
            <span className="font-bold text-2xl text-white">
              VCH<span className="font-light text-[#a8a8a8]">CLOUD</span>
            </span>
          </Link>
          <p className="text-[#a8a8a8] mt-2">Sign in to your dashboard</p>
        </div>

        {/* Form Card */}
        <div className="bg-[#262626] border border-[#393939] rounded-lg p-8">
          {/* Alert */}
          {alert.show && (
            <div
              className={`mb-6 p-4 rounded border ${
                alert.type === 'success'
                  ? 'bg-[#defbe6] border-[#24a148] text-[#0e6027]'
                  : 'bg-[#fff1f1] border-[#da1e28] text-[#750e13]'
              }`}
            >
              {alert.message}
            </div>
          )}

          <form onSubmit={handleSubmit} className="space-y-6">
            {/* Username/Email Field */}
            <div>
              <label htmlFor="login" className="block text-sm font-medium text-[#c6c6c6] mb-2">
                Username or Email
              </label>
              <input
                type="text"
                id="login"
                name="login"
                value={formData.login}
                onChange={handleChange}
                required
                className="w-full px-4 py-3 bg-[#161616] border border-[#393939] text-white rounded focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] outline-none transition placeholder-[#6f6f6f]"
                placeholder="Enter your username or email"
              />
            </div>

            {/* Password Field */}
            <div>
              <label htmlFor="password" className="block text-sm font-medium text-[#c6c6c6] mb-2">
                Password
              </label>
              <input
                type="password"
                id="password"
                name="password"
                value={formData.password}
                onChange={handleChange}
                required
                className="w-full px-4 py-3 bg-[#161616] border border-[#393939] text-white rounded focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] outline-none transition placeholder-[#6f6f6f]"
                placeholder="Enter your password"
              />
            </div>

            {/* Submit Button */}
            <button
              type="submit"
              disabled={loading}
              className="w-full py-4 bg-[#0f62fe] text-white font-bold rounded hover:bg-[#0353e9] transition flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {loading ? (
                <>
                  <FontAwesomeIcon icon={faSpinner} className="animate-spin" />
                  Processing...
                </>
              ) : (
                <>
                  Sign In
                  <FontAwesomeIcon icon={faArrowRight} />
                </>
              )}
            </button>
          </form>

          {/* Register Link */}
          <p className="text-center text-[#a8a8a8] mt-6">
            Don't have an account?{' '}
            <Link to="/register" className="text-[#0f62fe] font-medium hover:underline">
              Create one
            </Link>
          </p>
        </div>
      </div>
    </div>
  );
};

export default Login;
