import { useState, useEffect } from 'react';
import { Link, useSearchParams } from 'react-router-dom';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import {
  faMagnifyingGlass,
  faLayerGroup,
  faDesktop,
  faHandshake,
  faKey,
  faArrowUpRightDots,
  faShieldCat,
  faShieldVirus
} from '@fortawesome/free-solid-svg-icons';
import { faWindows } from '@fortawesome/free-brands-svg-icons';
import Navbar from '../components/Navbar';
import Footer from '../components/Footer';

const Community = () => {
  const [searchParams] = useSearchParams();
  const [searchQuery, setSearchQuery] = useState(searchParams.get('q') || '');
  const [filteredArticles, setFilteredArticles] = useState([]);

  const articles = [
    {
      id: 1,
      title: 'First RDP Login Guide',
      description: 'New to HostCraft? Learn how to access your Windows VPS, understand the panel, and set up your first instance.',
      category: 'Getting Started',
      icon: faDesktop,
      color: '#0f62fe',
      href: '/community/first-rdp-login',
      keywords: 'windows connect remote desktop rdp'
    },
    {
      id: 2,
      title: 'Service Level Agreement',
      description: 'Understand our 99.9% uptime guarantees and compensation policies for network and hardware availability.',
      category: 'Getting Started',
      icon: faHandshake,
      color: '#0f62fe',
      href: '/community/sla',
      keywords: 'sla uptime'
    },
    {
      id: 3,
      title: 'Reset Root Password',
      description: 'Reinstall OS, change passwords, resize disks, and manage snapshots via Virtualizor.',
      category: 'VPS Management',
      icon: faKey,
      color: '#8a3ffc',
      href: '/community/reset-root-password',
      keywords: 'reset root password administrator linux windows'
    },
    {
      id: 4,
      title: 'VNC Console Access',
      description: 'Locked out? Use the emergency VNC console in the dashboard to regain access to your server.',
      category: 'VPS Management',
      icon: faDesktop,
      color: '#8a3ffc',
      href: '/community/vnc-console',
      keywords: 'vnc console access remote viewer'
    },
    {
      id: 5,
      title: 'Upgrade Resources',
      description: 'Step-by-step guide to upgrading your CPU, RAM, and NVMe Disk space instantly without downtime.',
      category: 'Billing',
      icon: faArrowUpRightDots,
      color: '#ba4e00',
      href: '/community/upgrade-resources',
      keywords: 'upgrade vps resize disk cpu ram'
    },
    {
      id: 6,
      title: 'Path.net DDoS Filters',
      description: 'Configure firewalls, understand anti-DDoS filters, and secure your remote desktop sessions.',
      category: 'Security',
      icon: faShieldCat,
      color: '#198038',
      href: '/community/ddos-filters',
      keywords: 'path.net ddos filters game port firewall'
    },
    {
      id: 7,
      title: 'Windows Firewall Basics',
      description: 'A beginner guide to the Windows Defender Firewall. Learn how to open ports for your game servers.',
      category: 'Security',
      icon: faWindows,
      color: '#198038',
      href: '/community/windows-firewall',
      keywords: 'windows firewall defender security'
    },
    {
      id: 8,
      title: 'Configure UFW (Linux)',
      description: 'Best practices for securing your server ports using UFW (Ubuntu/Debian).',
      category: 'Security',
      icon: faShieldVirus,
      color: '#198038',
      href: '/community/configure-ufw',
      keywords: 'configure firewall ufw iptables'
    }
  ];

  useEffect(() => {
    filterArticles(searchQuery);
  }, [searchQuery]);

  const filterArticles = (query) => {
    const term = query.toLowerCase().trim();
    if (!term) {
      setFilteredArticles(articles);
      return;
    }

    const filtered = articles.filter((article) => {
      const searchText = `${article.title} ${article.description} ${article.category} ${article.keywords}`.toLowerCase();
      return searchText.includes(term);
    });

    setFilteredArticles(filtered);
  };

  const getCategoryColor = (category) => {
    switch (category) {
      case 'Getting Started':
        return { bg: 'bg-[#0f62fe]/10', text: 'text-[#0f62fe]', hover: 'hover:border-[#0f62fe]' };
      case 'VPS Management':
        return { bg: 'bg-[#8a3ffc]/10', text: 'text-[#8a3ffc]', hover: 'hover:border-[#8a3ffc]' };
      case 'Security':
        return { bg: 'bg-[#198038]/10', text: 'text-[#198038]', hover: 'hover:border-[#198038]' };
      case 'Billing':
        return { bg: 'bg-[#ba4e00]/10', text: 'text-[#ba4e00]', hover: 'hover:border-[#ba4e00]' };
      default:
        return { bg: 'bg-[#6f6f6f]/10', text: 'text-[#6f6f6f]', hover: 'hover:border-[#6f6f6f]' };
    }
  };

  return (
    <div className="pt-14 bg-[#f4f4f4] min-h-screen">
      {/* Hero Section */}
      <section className="bg-[#161616] py-20 border-b border-[#393939] relative overflow-hidden">
        {/* Technical Grid Pattern */}
        <div
          className="absolute inset-0 z-0 opacity-10 pointer-events-none"
          style={{
            backgroundImage: 'linear-gradient(#ffffff 1px, transparent 1px), linear-gradient(to right, #ffffff 1px, transparent 1px)',
            backgroundSize: '40px 40px'
          }}
        ></div>

        <div className="bx--grid relative z-10">
          <div className="max-w-4xl mx-auto text-center">
            <div className="inline-flex items-center gap-2 px-3 py-1 bg-[#0f62fe]/10 border border-[#0f62fe] text-[#0f62fe] text-xs font-bold uppercase tracking-wider mb-6 rounded-full">
              <FontAwesomeIcon icon={faLayerGroup} />
              Knowledge Hub
            </div>

            <h1 className="text-4xl md:text-5xl font-light text-white mb-8 tracking-tight">
              Browse All <span className="font-bold text-transparent bg-clip-text bg-gradient-to-r from-[#0f62fe] to-[#8a3ffc]">Guides</span>
            </h1>

            {/* Search Input */}
            <div className="relative max-w-2xl mx-auto group">
              <div className="absolute inset-y-0 left-0 pl-6 flex items-center pointer-events-none">
                <FontAwesomeIcon icon={faMagnifyingGlass} className="text-[#6f6f6f] group-focus-within:text-[#0f62fe] transition-colors" />
              </div>
              <input
                type="text"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                placeholder="Filter by keyword (e.g., 'Root', 'Firewall', 'Billing')..."
                className="w-full pl-14 pr-4 py-4 bg-[#262626] border border-[#393939] text-white rounded-lg focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] outline-none transition placeholder-[#6f6f6f] shadow-2xl"
              />
            </div>

            <p className="text-[#6f6f6f] text-sm mt-4">
              Showing <span className="text-white font-mono font-bold">{filteredArticles.length}</span> articles available.
            </p>
          </div>
        </div>
      </section>

      {/* Articles List */}
      <section className="py-16 bg-white border-b border-[#e0e0e0] min-h-[60vh]">
        <div className="bx--grid">
          <div className="max-w-4xl mx-auto">
            {filteredArticles.length > 0 ? (
              <div className="grid gap-6">
                {filteredArticles.map((article) => {
                  const categoryStyle = getCategoryColor(article.category);
                  return (
                    <Link
                      key={article.id}
                      to={article.href}
                      className={`group block p-6 border border-[#e0e0e0] rounded-xl ${categoryStyle.hover} hover:shadow-lg transition-all duration-300 bg-white relative overflow-hidden`}
                    >
                      <div
                        className="absolute left-0 top-0 bottom-0 w-1 opacity-0 group-hover:opacity-100 transition-opacity"
                        style={{ backgroundColor: article.color }}
                      ></div>
                      <div className="flex items-start gap-6">
                        <div
                          className="w-12 h-12 rounded-lg flex items-center justify-center flex-shrink-0 transition"
                          style={{
                            backgroundColor: `${article.color}10`,
                            color: article.color
                          }}
                        >
                          <FontAwesomeIcon icon={article.icon} className="text-xl" />
                        </div>
                        <div className="flex-grow">
                          <div className="flex justify-between items-start flex-wrap gap-2">
                            <h3
                              className="text-xl font-bold text-[#161616] mb-2 group-hover:transition"
                              style={{ '--hover-color': article.color }}
                            >
                              {article.title}
                            </h3>
                            <span
                              className={`text-[10px] font-mono font-bold uppercase tracking-wider px-2 py-1 rounded ${categoryStyle.bg} ${categoryStyle.text}`}
                            >
                              {article.category}
                            </span>
                          </div>
                          <p className="text-[#525252] text-sm leading-relaxed max-w-2xl">
                            {article.description}
                          </p>
                        </div>
                      </div>
                    </Link>
                  );
                })}
              </div>
            ) : (
              <div className="text-center py-16">
                <div className="inline-flex justify-center items-center w-16 h-16 bg-[#f4f4f4] rounded-full mb-4">
                  <FontAwesomeIcon icon={faMagnifyingGlass} className="text-[#a8a8a8] text-2xl" />
                </div>
                <h3 className="text-lg font-bold text-[#161616]">No articles found</h3>
                <p className="text-[#6f6f6f] mb-6">Try adjusting your search terms.</p>
                <button
                  onClick={() => setSearchQuery('')}
                  className="text-[#0f62fe] font-bold hover:underline text-sm uppercase tracking-wider"
                >
                  Clear Search
                </button>
              </div>
            )}
          </div>
        </div>
      </section>
    </div>
  );
};

export default Community;
