import { useState } from 'react';
import { Link } from 'react-router-dom';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import {
  faArrowLeft,
  faPaperPlane,
  faSpinner,
  faChevronDown
} from '@fortawesome/free-solid-svg-icons';
import Navbar from '../components/Navbar';
import Footer from '../components/Footer';

const SupportTicket = () => {
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    department: 'technical',
    priority: 'medium',
    subject: '',
    message: ''
  });
  const [loading, setLoading] = useState(false);
  const [alert, setAlert] = useState({ show: false, type: '', message: '' });

  const handleChange = (e) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value
    });
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    setAlert({ show: false, type: '', message: '' });

    // Simulate API call
    setTimeout(() => {
      setAlert({
        show: true,
        type: 'success',
        message: 'Ticket submitted successfully! You will receive a confirmation email shortly.'
      });
      setFormData({
        name: '',
        email: '',
        department: 'technical',
        priority: 'medium',
        subject: '',
        message: ''
      });
      setLoading(false);
    }, 1500);
  };

  return (
    <div className="pt-14 bg-[#f4f4f4] min-h-screen">
      <main className="py-16 bx--grid">
        <div className="max-w-2xl mx-auto">
          {/* Breadcrumb */}
          <Link
            to="/support"
            className="inline-flex items-center gap-2 text-[#6f6f6f] hover:text-[#0f62fe] mb-8 font-mono text-sm transition"
          >
            <FontAwesomeIcon icon={faArrowLeft} />
            Back to Support Center
          </Link>

          <div className="bg-white border border-[#e0e0e0] rounded-xl p-8 md:p-10 shadow-sm relative overflow-hidden">
            {/* Header */}
            <div className="mb-8">
              <h1 className="text-3xl font-light text-[#161616] mb-2">Open New Ticket</h1>
              <p className="text-[#6f6f6f]">
                Please provide detailed information so we can assist you faster.
              </p>
            </div>

            {/* Alert */}
            {alert.show && (
              <div
                className={`mb-6 p-4 rounded border ${
                  alert.type === 'success'
                    ? 'bg-[#defbe6] border-[#24a148] text-[#0e6027]'
                    : 'bg-[#fff1f1] border-[#da1e28] text-[#750e13]'
                }`}
              >
                {alert.message}
              </div>
            )}

            <form onSubmit={handleSubmit} className="space-y-6">
              {/* Row 1: Name & Email */}
              <div className="grid md:grid-cols-2 gap-6">
                <div className="space-y-2">
                  <label htmlFor="name" className="block text-sm font-bold text-[#161616]">
                    Full Name
                  </label>
                  <input
                    type="text"
                    id="name"
                    name="name"
                    value={formData.name}
                    onChange={handleChange}
                    required
                    className="w-full bg-[#f4f4f4] border border-[#e0e0e0] rounded p-3 outline-none focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] transition text-[#161616]"
                    placeholder="John Doe"
                  />
                </div>
                <div className="space-y-2">
                  <label htmlFor="email" className="block text-sm font-bold text-[#161616]">
                    Email Address
                  </label>
                  <input
                    type="email"
                    id="email"
                    name="email"
                    value={formData.email}
                    onChange={handleChange}
                    required
                    className="w-full bg-[#f4f4f4] border border-[#e0e0e0] rounded p-3 outline-none focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] transition text-[#161616]"
                    placeholder="john@example.com"
                  />
                </div>
              </div>

              {/* Row 2: Department & Priority */}
              <div className="grid md:grid-cols-2 gap-6">
                <div className="space-y-2">
                  <label htmlFor="department" className="block text-sm font-bold text-[#161616]">
                    Department
                  </label>
                  <div className="relative">
                    <select
                      id="department"
                      name="department"
                      value={formData.department}
                      onChange={handleChange}
                      className="w-full bg-[#f4f4f4] border border-[#e0e0e0] rounded p-3 outline-none focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] transition text-[#161616] appearance-none cursor-pointer"
                    >
                      <option value="technical">Technical Support</option>
                      <option value="billing">Billing & Sales</option>
                      <option value="abuse">Abuse Report</option>
                    </select>
                    <div className="absolute inset-y-0 right-0 flex items-center px-4 pointer-events-none text-[#6f6f6f]">
                      <FontAwesomeIcon icon={faChevronDown} className="text-xs" />
                    </div>
                  </div>
                </div>
                <div className="space-y-2">
                  <label htmlFor="priority" className="block text-sm font-bold text-[#161616]">
                    Priority
                  </label>
                  <div className="relative">
                    <select
                      id="priority"
                      name="priority"
                      value={formData.priority}
                      onChange={handleChange}
                      className="w-full bg-[#f4f4f4] border border-[#e0e0e0] rounded p-3 outline-none focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] transition text-[#161616] appearance-none cursor-pointer"
                    >
                      <option value="low">Low</option>
                      <option value="medium">Medium</option>
                      <option value="high">High (Production Down)</option>
                    </select>
                    <div className="absolute inset-y-0 right-0 flex items-center px-4 pointer-events-none text-[#6f6f6f]">
                      <FontAwesomeIcon icon={faChevronDown} className="text-xs" />
                    </div>
                  </div>
                </div>
              </div>

              {/* Subject */}
              <div className="space-y-2">
                <label htmlFor="subject" className="block text-sm font-bold text-[#161616]">
                  Subject
                </label>
                <input
                  type="text"
                  id="subject"
                  name="subject"
                  value={formData.subject}
                  onChange={handleChange}
                  required
                  className="w-full bg-[#f4f4f4] border border-[#e0e0e0] rounded p-3 outline-none focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] transition text-[#161616]"
                  placeholder="e.g. Cannot connect to RDP"
                />
              </div>

              {/* Message */}
              <div className="space-y-2">
                <label htmlFor="message" className="block text-sm font-bold text-[#161616]">
                  Message
                </label>
                <textarea
                  id="message"
                  name="message"
                  rows={6}
                  value={formData.message}
                  onChange={handleChange}
                  required
                  className="w-full bg-[#f4f4f4] border border-[#e0e0e0] rounded p-3 outline-none focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] transition text-[#161616] resize-y"
                  placeholder="Describe your issue in detail. Please include any relevant logs or error messages."
                ></textarea>
              </div>

              {/* Submit Button */}
              <button
                type="submit"
                disabled={loading}
                className="w-full py-4 bg-[#0f62fe] hover:bg-[#0353e9] text-white font-bold rounded transition shadow-lg shadow-[#0f62fe]/10 flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {loading ? (
                  <>
                    <FontAwesomeIcon icon={faSpinner} className="animate-spin" />
                    Submitting...
                  </>
                ) : (
                  <>
                    <span>Submit Ticket</span>
                    <FontAwesomeIcon icon={faPaperPlane} />
                  </>
                )}
              </button>
            </form>
          </div>

          {/* Footer Note */}
          <p className="text-center text-xs text-[#a8a8a8] mt-6">
            By submitting this ticket, you agree to our{' '}
            <Link to="/legal/terms" className="underline hover:text-[#0f62fe]">
              Terms of Service
            </Link>
            . <br />
            Our average response time is currently{' '}
            <span className="font-bold text-[#0f62fe]">15 minutes</span>.
          </p>
        </div>
      </main>
    </div>
  );
};

export default SupportTicket;
