import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import {
  faMicrochip,
  faHardDrive,
  faShareNodes,
  faShieldCat,
  faServer,
  faGaugeHigh,
  faHdd,
  faNetworkWired,
  faDatabase,
  faChevronDown,
  faArrowRight,
  faCheck,
  faBolt,
  faShieldHalved,
  faSliders,
  faGlobe,
  faStar,
  faMagnifyingGlass,
  faBook,
  faCreditCard,
  faClock,
  faTicket
} from '@fortawesome/free-solid-svg-icons';
import { faCircle } from '@fortawesome/free-regular-svg-icons';
import VPSCard from '../components/VPSCard';

const Home = () => {
  const [activeTab, setActiveTab] = useState('compute');

  // Scroll reveal effect
  useEffect(() => {
    const handleScroll = () => {
      const reveals = document.querySelectorAll('.reveal');
      reveals.forEach((element) => {
        const windowHeight = window.innerHeight;
        const elementTop = element.getBoundingClientRect().top;
        const elementVisible = 150;

        if (elementTop < windowHeight - elementVisible) {
          element.classList.add('active');
        }
      });
    };

    window.addEventListener('scroll', handleScroll);
    handleScroll(); // Initial check

    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  const features = [
    {
      icon: faMicrochip,
      number: '01',
      title: 'Dual-Socket Beast',
      description: 'Fleet includes <strong>Dual-Socket Xeon Platinum</strong> and Gold processors. Each node carries <strong>2 CPUs</strong> (up to 104 cores) for redundant processing power.',
      footer: 'HARDWARE: 2x XEON PLATINUM',
      bgColor: 'bg-blue-5',
      textColor: 'text-blue-60',
      hoverBg: 'group-hover:bg-blue-60',
      hoverText: 'group-hover:text-blue-60',
      hoverNumber: 'group-hover:text-blue-10',
      topBorder: 'bg-blue-60'
    },
    {
      icon: faHardDrive,
      number: '02',
      title: 'Redundant Storage',
      description: 'Data is replicated across our NVMe cluster (RAID-10) with automated <strong>S3 Object Storage</strong> backups. Your files are safe from single-drive hardware failures.',
      footer: 'STORAGE: NVMe + S3',
      bgColor: 'bg-purple-10',
      textColor: 'text-purple-60',
      hoverBg: 'group-hover:bg-purple-60',
      hoverText: 'group-hover:text-purple-60',
      hoverNumber: 'group-hover:text-purple-10',
      topBorder: 'bg-purple-60'
    },
    {
      icon: faShareNodes,
      number: '03',
      title: 'Edge Optimization',
      description: 'Whether it\'s <strong>Bulk SMTP</strong> delivery or <strong>Edge CDN</strong> caching, our 10Gbps uplinks ensure your content reaches users instantly, worldwide.',
      footer: 'UPLINK: 10 GBPS FIBER',
      bgColor: 'bg-teal-10',
      textColor: 'text-teal-60',
      hoverBg: 'group-hover:bg-teal-60',
      hoverText: 'group-hover:text-teal-60',
      hoverNumber: 'group-hover:text-teal-10',
      topBorder: 'bg-teal-60'
    },
    {
      icon: faShieldCat,
      number: '04',
      title: 'Pathfinder DDoS',
      description: 'Always-on L3/L4/L7 protection. Our custom filters analyze traffic in real-time to block attacks without adding latency to legitimate players or users.',
      footer: 'FILTER: PATH.NET + COSMIC',
      bgColor: 'bg-green-5',
      textColor: 'text-green-50',
      hoverBg: 'group-hover:bg-green-50',
      hoverText: 'group-hover:text-green-50',
      hoverNumber: 'group-hover:text-green-10',
      topBorder: 'bg-green-50'
    }
  ];

  // Top 3 VPS Plans for cards display
  const topPlans = [
    {
      name: 'Starter VPS',
      code: 'NODE_CLUSTER_01',
      price: '$9.00',
      badge: 'Entry Point',
      badgeClass: 'bg-gray-10 text-gray-60',
      specs: [
        { label: 'vCores (Dual CPU)', value: '2 Cores' },
        { label: 'DDR4 RAM', value: '4 GB' },
        { label: 'NVMe Storage', value: '48 GB' }
      ],
      buttonText: 'Deploy Sandbox',
      featured: false
    },
    {
      name: 'Basic VPS',
      code: 'NODE_CLUSTER_02',
      price: '$13.50',
      badge: null,
      specs: [
        { label: 'vCores (Dual CPU)', value: '4 Cores' },
        { label: 'DDR4 RAM', value: '8 GB' },
        { label: 'NVMe Storage', value: '48 GB' },
        { label: 'Port Speed', value: '10 Gbps', highlight: true }
      ],
      buttonText: 'Deploy High Performance',
      featured: true
    },
    {
      name: 'Developer VPS',
      code: 'NODE_CLUSTER_03',
      price: '$22.50',
      badge: 'Powerhouse',
      badgeClass: 'bg-purple-10 text-purple-60',
      specs: [
        { label: 'vCores (Dual CPU)', value: '6 Cores' },
        { label: 'DDR4 RAM', value: '16 GB' },
        { label: 'NVMe Storage', value: '48 GB' }
      ],
      buttonText: 'Deploy Workstation',
      featured: false
    }
  ];

  // Full pricing table data
  const allPlans = [
    { name: 'Starter', cores: '2 Cores', ram: '4 GB', storage: '48 GB', price: '$9.00', hot: false },
    { name: 'Basic', cores: '4 Cores', ram: '8 GB', storage: '48 GB', price: '$13.50', hot: true },
    { name: 'Developer', cores: '6 Cores', ram: '16 GB', storage: '48 GB', price: '$22.50', hot: false },
    { name: 'Business', cores: '8 Cores', ram: '24 GB', storage: '48 GB', price: '$28.80', hot: false },
    { name: 'Pro', cores: '10 Cores', ram: '32 GB', storage: '48 GB', price: '$40.50', hot: false },
    { name: 'Elite', cores: '12 Cores', ram: '48 GB', storage: '48 GB', price: '$51.30', hot: false },
    { name: 'Enterprise', cores: '16 Cores', ram: '64 GB', storage: '48 GB', price: '$63.00', hot: false },
    { name: 'Titan', cores: '20 Cores', ram: '96 GB', storage: '48 GB', price: '$72.00', hot: false },
    { name: 'Omega', cores: '24 Cores', ram: '128 GB', storage: '48 GB', price: '$90.00', hot: false, highlight: true }
  ];

  // Infrastructure tabs
  const infraTabs = [
    {
      id: 'compute',
      icon: faMicrochip,
      title: 'Compute',
      subtitle: 'Dual-Socket Nodes',
      bgColor: 'bg-blue-10',
      textColor: 'text-blue-60',
      borderColor: 'border-l-blue-60',
      activeIconBg: 'bg-blue-60'
    },
    {
      id: 'storage',
      icon: faServer,
      title: 'Storage Cluster',
      subtitle: 'NVMe + Ceph S3',
      bgColor: 'bg-purple-10',
      textColor: 'text-purple-60',
      borderColor: 'border-l-purple-60',
      activeIconBg: 'bg-purple-60'
    },
    {
      id: 'security',
      icon: faShieldHalved,
      title: 'Security',
      subtitle: 'Path.net Filter',
      bgColor: 'bg-green-10',
      textColor: 'text-green-60',
      borderColor: 'border-l-green-60',
      activeIconBg: 'bg-green-60'
    },
    {
      id: 'control',
      icon: faSliders,
      title: 'Control',
      subtitle: 'Virtualizor & API',
      bgColor: 'bg-teal-10',
      textColor: 'text-teal-60',
      borderColor: 'border-l-teal-60',
      activeIconBg: 'bg-teal-60'
    },
    {
      id: 'network',
      icon: faGlobe,
      title: 'Network',
      subtitle: 'Global PoPs',
      bgColor: 'bg-orange-10',
      textColor: 'text-orange-60',
      borderColor: 'border-l-orange-60',
      activeIconBg: 'bg-orange-60'
    }
  ];

  // Comparison table data
  const comparisonRows = [
    {
      feature: 'CPU Disclosure',
      impact: 'Impacts: Stability under load',
      competitor: 'vCPU (Model Hidden)',
      vchcloud: 'Dual Xeon Platinum/Gold'
    },
    {
      feature: 'Storage Tech',
      impact: 'Impacts: RDP login speed & boot time',
      competitor: 'SATA SSD / Mixed',
      vchcloud: 'Enterprise NVMe (All Plans)'
    },
    {
      feature: 'Network Capacity',
      impact: 'Impacts: Latency & smoothness',
      competitor: '1 Gbps Shared',
      vchcloud: '10 Gbps Port'
    },
    {
      feature: 'Windows RDP',
      impact: 'Impacts: Remote work capability',
      competitor: 'Extra Cost / Limited',
      vchcloud: 'Included & Optimized'
    },
    {
      feature: 'Deployment',
      impact: 'Impacts: Time to start working',
      competitor: 'Manual / Delayed',
      vchcloud: 'Fast Automated Setup'
    }
  ];


  return (
    <div className="pt-14">
      {/* HERO SECTION - Light Background like original */}
      <header className="pt-32 pb-20 border-b border-[#e0e0e0] relative overflow-hidden">
        {/* Background Image */}
        <div
          className="absolute inset-0 z-0"
          style={{
            backgroundImage: 'url(https://iceline-hosting.com/_astro/minecraft-bg._2Yg98vP_1NkiVz.webp)',
            backgroundSize: 'cover',
            backgroundPosition: 'center',
            backgroundRepeat: 'no-repeat'
          }}
        ></div>

        {/* Dark Overlay */}
        <div className="absolute inset-0 z-0 bg-black/70"></div>

        {/* Technical Grid Background */}
        {/* <div
          className="absolute inset-0 z-0 opacity-10 pointer-events-none"
          style={{
            backgroundImage: 'linear-gradient(#ffffff 1px, transparent 1px), linear-gradient(to right, #ffffff 1px, transparent 1px)',
            backgroundSize: '50px 50px'
          }}
        ></div> */}

        <div className="bx--grid relative z-10">
          <div className="grid grid-cols-1 md:grid-cols-12 gap-12 items-center">
            {/* Left: Text Content */}
            <div className="md:col-span-7 reveal active">
              {/* Status Badge */}
              <div className="inline-flex items-center gap-2 px-3 py-1.5 bg-black/50 backdrop-blur-sm border border-white/20 text-xs font-mono text-[#78a9ff] mb-6 rounded">
                <span className="relative flex h-2 w-2">
                  <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-[#42be65] opacity-75"></span>
                  <span className="relative inline-flex rounded-full h-2 w-2 bg-[#24a148]"></span>
                </span>
                DUAL-SOCKET CLUSTER ACTIVE
              </div>

              {/* Gradient Glow - Decorative */}
              <div className="absolute top-0 right-0 w-64 h-64 bg-[#0f62fe]/20 blur-[100px] pointer-events-none"></div>

              {/* Main Headline */}
              <h1 className="text-5xl md:text-7xl font-light text-white mb-6 leading-[1.1] tracking-tight">
                Dual-Socket Power.<br />
                <span className="font-semibold text-[#a56eff]">Cloud Reliability.</span>
              </h1>

              {/* Subheadline */}
              <p className="text-xl text-gray-300 mb-10 max-w-xl font-light leading-relaxed">
                Deploy <strong className="text-white">High Performance VPS RDP</strong>, High-Availability Cloud,
                and <strong className="text-white">S3 Object Storage</strong> on our redundant cluster infrastructure.
              </p>

              {/* Dual CTA Buttons */}
              <div className="flex flex-col sm:flex-row gap-0 border border-white/20 w-fit shadow-xl rounded overflow-hidden">
                <Link
                  to="/pricing/vps"
                  className="px-8 py-4 bg-[#0f62fe] text-white font-medium hover:bg-[#0043ce] transition flex items-center gap-4 group"
                >
                  View Plans
                  <FontAwesomeIcon icon={faArrowRight} className="group-hover:translate-x-1 transition" />
                </Link>
                <a
                  href="#features"
                  className="px-8 py-4 bg-white/10 backdrop-blur-sm text-white font-medium hover:bg-white/20 transition flex items-center gap-4 border-t sm:border-t-0 sm:border-l border-white/20"
                >
                  View Features
                  <FontAwesomeIcon icon={faServer} className="text-gray-400" />
                </a>
              </div>

              {/* Tech Specs Row */}
              <div className="mt-12 flex flex-wrap gap-6 md:gap-8 text-xs text-gray-400 font-mono">
                <div className="flex items-center gap-2">
                  <FontAwesomeIcon icon={faMicrochip} className="text-[#0f62fe]" />
                  UP TO 104 CORES
                </div>
                <div className="flex items-center gap-2">
                  <FontAwesomeIcon icon={faNetworkWired} className="text-[#0f62fe]" />
                  CLUSTERED NETWORK
                </div>
                <div className="flex items-center gap-2">
                  <FontAwesomeIcon icon={faDatabase} className="text-[#0f62fe]" />
                  S3 OBJECT STORAGE
                </div>
              </div>
            </div>

            {/* Right: VPS Card */}
            <div className="md:col-span-5 reveal" style={{ animationDelay: '0.2s' }}>
              <VPSCard />
            </div>
          </div>
        </div>
      </header>

      {/* TECHNOLOGY & FEATURES STACK */}
      <section id="features" className="py-24 bg-white border-b border-[#e0e0e0] relative overflow-hidden">
        {/* Technical Background Pattern */}
        <div
          className="absolute inset-0 z-0 opacity-[0.03] pointer-events-none"
          style={{
            backgroundImage: 'radial-gradient(#000 1px, transparent 1px)',
            backgroundSize: '24px 24px'
          }}
        ></div>

        <div className="bx--grid relative z-10">
          {/* Section Header */}
          <div className="flex flex-col md:flex-row justify-between items-end mb-16 reveal">
            <div className="max-w-2xl">
              <h2 className="text-3xl md:text-4xl font-light text-[#161616] mb-4">
                The <span className="font-semibold">VCHCLOUD</span> Standard.
              </h2>
              <p className="text-[#6f6f6f] text-lg font-light leading-relaxed">
                We don't oversell single-socket consumer boards. We deploy enterprise-grade
                <strong> Dual-Socket</strong> Intel® hardware in a highly available cluster,
                ensuring massive parallel processing power.
              </p>
            </div>
            {/* Metric Box */}
            <div className="hidden md:block text-right bg-[#f9f9f9] p-4 border border-[#e0e0e0] rounded-lg">
              <div className="flex items-center justify-end gap-2 mb-1">
                <span className="relative flex h-2 w-2">
                  <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-[#42be65] opacity-75"></span>
                  <span className="relative inline-flex rounded-full h-2 w-2 bg-[#24a148]"></span>
                </span>
                <div className="text-xs font-bold text-[#8d8d8d] uppercase tracking-widest">Network Uptime</div>
              </div>
              <div className="text-4xl font-mono text-[#161616] tracking-tighter">
                99.99<span className="text-[#a8a8a8] text-2xl">%</span>
              </div>
            </div>
          </div>

          {/* Features Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-px bg-gray-20 border border-gray-20 reveal shadow-sm">
            {features.map((feature, index) => (
              <div
                key={index}
                className="group relative bg-white p-8 hover:bg-gray-5 transition duration-300"
              >
                {/* Top border on hover */}
                <div className={`absolute top-0 left-0 w-full h-1 ${feature.topBorder} transform scale-x-0 group-hover:scale-x-100 transition duration-300 origin-left`}></div>

                <div className="flex justify-between items-start mb-6">
                  <div className={`w-12 h-12 ${feature.bgColor} ${feature.textColor} flex items-center justify-center rounded-lg ${feature.hoverBg} group-hover:text-white transition-colors`}>
                    <FontAwesomeIcon icon={feature.icon} className="text-xl" />
                  </div>
                  <span className={`text-4xl font-light text-gray-20 ${feature.hoverNumber} transition-colors select-none`}>
                    {feature.number}
                  </span>
                </div>

                <h3 className={`text-lg font-bold text-gray-100 mb-3 ${feature.hoverText} transition-colors`}>
                  {feature.title}
                </h3>
                <p
                  className="text-sm text-gray-60 leading-relaxed mb-6"
                  dangerouslySetInnerHTML={{ __html: feature.description }}
                />

                <div className="border-t border-gray-10 pt-4">
                  <span className="text-[10px] font-mono text-gray-40 uppercase tracking-wider">
                    {feature.footer}
                  </span>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* VPS PRICING */}
      <section id="pricing" className="py-24 bg-white border-b border-gray-20 relative overflow-hidden">
        {/* Subtle Grid Background */}
        <div
          className="absolute inset-0 opacity-[0.03] pointer-events-none"
          style={{
            backgroundImage: 'linear-gradient(#000 1px, transparent 1px), linear-gradient(to right, #000 1px, transparent 1px)',
            backgroundSize: '40px 40px'
          }}
        ></div>

        <div className="bx--grid relative z-10">
          {/* Header */}
          <div className="flex flex-col md:flex-row justify-between items-end mb-16 border-b border-gray-20 pb-6 reveal">
            <div>
              <h2 className="text-3xl md:text-4xl font-light text-gray-100 mb-2">
                High Performance <span className="font-semibold">Compute</span>
              </h2>
              <p className="text-gray-70 text-lg font-light">
                Powered by <strong>Dual-Socket Xeon Platinum</strong>. 100% NVMe. No lock-ins.
              </p>
            </div>
            {/* Toggle */}
            <div className="hidden md:flex items-center gap-3 bg-gray-5 p-1 rounded-lg border border-gray-20">
              <button className="px-4 py-2 bg-white text-gray-100 shadow-sm text-xs font-bold rounded-md border border-gray-10">
                Monthly Billing
              </button>
              <button className="px-4 py-2 text-gray-50 text-xs font-bold hover:text-gray-80 transition">
                Hourly (API)
              </button>
            </div>
          </div>

          {/* TOP SELLERS GRID */}
          <div className="grid md:grid-cols-3 gap-8 reveal mb-20">
            {topPlans.map((plan, index) => (
              plan.featured ? (
                /* Featured Plan - Basic VPS */
                <div
                  key={index}
                  className="relative bg-gray-100 text-white border border-gray-100 shadow-2xl flex flex-col p-8 transform md:-translate-y-4 z-10"
                >
                  <div className="absolute top-0 inset-x-0 h-1 bg-gradient-to-r from-blue-60 to-purple-60"></div>
                  <div className="absolute top-4 right-4 text-blue-40 text-2xl animate-pulse">
                    <FontAwesomeIcon icon={faBolt} />
                  </div>

                  <h3 className="text-xl font-bold text-white">{plan.name}</h3>
                  <p className="text-xs text-gray-40 mb-6 font-mono">{plan.code}</p>

                  <div className="flex items-baseline gap-1 mb-8">
                    <span className="text-5xl font-light tracking-tighter text-white">{plan.price}</span>
                    <span className="text-sm text-gray-40">/mo</span>
                  </div>

                  <div className="space-y-4 mb-8 flex-grow">
                    {plan.specs.map((spec, idx) => (
                      <div key={idx} className="flex justify-between items-center border-b border-gray-80 pb-2">
                        <span className="text-sm text-gray-30">{spec.label}</span>
                        <span className={`font-mono font-bold text-lg ${spec.highlight ? 'text-blue-40' : 'text-white'}`}>
                          {spec.value}
                        </span>
                      </div>
                    ))}
                  </div>

                  <Link
                    to="/pricing/vps"
                    className="w-full py-4 bg-blue-60 text-white font-bold text-sm text-center hover:bg-blue-70 transition-colors shadow-lg shadow-blue-60/20"
                  >
                    {plan.buttonText}
                  </Link>
                </div>
              ) : (
                /* Regular Plan Cards */
                <div
                  key={index}
                  className={`group relative bg-white border border-gray-20 ${index === 2 ? 'hover:border-purple-60' : 'hover:border-blue-60'} transition-all duration-300 hover:-translate-y-1 hover:shadow-xl flex flex-col p-8`}
                >
                  {plan.badge && (
                    <div className={`absolute top-0 right-0 ${plan.badgeClass} text-[10px] font-bold px-3 py-1 uppercase tracking-wider`}>
                      {plan.badge}
                    </div>
                  )}

                  <h3 className="text-xl font-bold text-gray-100">{plan.name}</h3>
                  <p className="text-xs text-gray-50 mb-6 font-mono">{plan.code}</p>

                  <div className="flex items-baseline gap-1 mb-8">
                    <span className="text-4xl font-light tracking-tighter text-gray-100">{plan.price}</span>
                    <span className="text-sm text-gray-50">/mo</span>
                  </div>

                  <div className="space-y-4 mb-8 flex-grow">
                    {plan.specs.map((spec, idx) => (
                      <div key={idx} className="flex justify-between items-center border-b border-gray-10 pb-2">
                        <span className="text-sm text-gray-60">{spec.label}</span>
                        <span className="font-mono font-bold text-gray-100">{spec.value}</span>
                      </div>
                    ))}
                  </div>

                  <Link
                    to="/pricing/vps"
                    className="w-full py-3 border border-gray-20 text-gray-100 font-bold text-sm text-center hover:bg-gray-100 hover:text-white hover:border-gray-100 transition-colors"
                  >
                    {plan.buttonText}
                  </Link>
                </div>
              )
            ))}
          </div>

          {/* FULL TABLE */}
          <div className="reveal">
            <div className="flex items-center gap-4 mb-8">
              <h3 className="text-xl font-bold text-gray-100">Complete Resource List</h3>
              <div className="h-px flex-grow bg-gray-20"></div>
            </div>

            <div className="overflow-x-auto bg-white border border-gray-20 rounded-lg shadow-sm">
              <table className="w-full text-left text-sm">
                <thead>
                  <tr className="bg-gray-5 text-gray-50 text-xs uppercase tracking-wider font-mono border-b border-gray-20">
                    <th className="p-5 font-medium">Instance Type</th>
                    <th className="p-5 font-medium">vCPU (Dual Xeon)</th>
                    <th className="p-5 font-medium">RAM (DDR4)</th>
                    <th className="p-5 font-medium">Storage (NVMe)</th>
                    <th className="p-5 font-medium">Monthly</th>
                    <th className="p-5 font-medium text-right">Deploy</th>
                  </tr>
                </thead>
                <tbody className="divide-y divide-gray-10 font-mono text-gray-80">
                  {allPlans.map((plan, index) => (
                    <tr
                      key={index}
                      className={`hover:bg-blue-5/30 transition-colors group ${plan.hot ? 'bg-blue-5/10' : ''}`}
                    >
                      <td className="p-5 font-sans font-bold text-gray-100 flex items-center gap-2">
                        {plan.name}
                        {plan.hot && (
                          <span className="text-[10px] bg-blue-60 text-white px-1.5 rounded">HOT</span>
                        )}
                      </td>
                      <td className={`p-5 ${plan.hot ? 'font-bold' : ''} ${plan.highlight ? 'text-purple-60 font-bold' : ''}`}>
                        {plan.cores}
                      </td>
                      <td className={`p-5 ${plan.hot ? 'font-bold' : ''} ${plan.highlight ? 'text-purple-60 font-bold' : ''}`}>
                        {plan.ram}
                      </td>
                      <td className="p-5">{plan.storage}</td>
                      <td className="p-5 font-bold text-blue-60">{plan.price}</td>
                      <td className="p-5 text-right">
                        <Link
                          to="/pricing/vps"
                          className="text-gray-40 group-hover:text-blue-60 transition-all duration-300 inline-block group-hover:translate-x-1"
                        >
                          <FontAwesomeIcon icon={faArrowRight} />
                        </Link>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </div>
        </div>
      </section>

      {/* INFRASTRUCTURE EXPLORER */}
      <section id="infrastructure" className="py-24 bg-white border-b border-gray-20">
        <div className="bx--grid">
          <div className="mb-12 reveal">
            <h2 className="text-3xl font-light text-gray-100">The Hardware Stack</h2>
            <div className="h-1 w-20 bg-blue-60 mt-4"></div>
          </div>

          <div className="flex flex-col lg:flex-row border border-gray-20 bg-gray-5 shadow-2xl reveal rounded-xl overflow-hidden">
            {/* Left: Navigation Tabs */}
            <div className="lg:w-1/4 flex flex-row lg:flex-col border-b lg:border-b-0 lg:border-r border-gray-20 overflow-x-auto lg:overflow-hidden bg-gray-5">
              {infraTabs.map((tab) => (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id)}
                  className={`w-full text-left px-8 py-6 border-b border-gray-20 hover:bg-white/50 transition-all duration-300 flex items-center gap-4 border-l-4 outline-none group focus:outline-none ${
                    activeTab === tab.id ? `${tab.borderColor}` : 'border-l-transparent'
                  }`}
                >
                  <div
                    className={`w-10 h-10 rounded-lg flex items-center justify-center transition-colors ${
                      activeTab === tab.id
                        ? `${tab.activeIconBg} text-white`
                        : `${tab.bgColor} ${tab.textColor}`
                    }`}
                  >
                    <FontAwesomeIcon icon={tab.icon} className="text-lg" />
                  </div>
                  <div>
                    <span className={`block font-bold text-sm transition-colors ${
                      activeTab === tab.id ? tab.textColor : 'text-gray-100'
                    }`}>
                      {tab.title}
                    </span>
                    <span className="block text-xs text-gray-50 mt-1">{tab.subtitle}</span>
                  </div>
                </button>
              ))}
            </div>

            {/* Right: Content Panel */}
            <div className="lg:w-3/4 bg-white p-8 lg:p-12">
              {activeTab === 'compute' && (
                <div>
                  <h3 className="text-2xl font-bold text-gray-100 mb-4">Dual-Socket Architecture</h3>
                  <p className="text-gray-60 leading-relaxed mb-6">
                    Each physical server in our cluster is equipped with <strong>two Intel Xeon Platinum or Gold processors</strong>,
                    providing up to 104 cores per node. This dual-socket design ensures redundancy and massive parallel processing capability.
                  </p>
                  <div className="flex flex-wrap gap-3">
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">2x Xeon Platinum</span>
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">Up to 104 Cores</span>
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">DDR4 ECC RAM</span>
                  </div>
                </div>
              )}
              {activeTab === 'storage' && (
                <div>
                  <h3 className="text-2xl font-bold text-gray-100 mb-4">NVMe + Ceph S3 Storage</h3>
                  <p className="text-gray-60 leading-relaxed mb-6">
                    All VPS storage runs on enterprise NVMe drives in RAID-10 configuration. Additionally, our Ceph S3-compatible
                    object storage cluster provides scalable backup and static asset hosting.
                  </p>
                  <div className="flex flex-wrap gap-3">
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">NVMe RAID-10</span>
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">Ceph S3</span>
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">3.5GB/s Read</span>
                  </div>
                </div>
              )}
              {activeTab === 'security' && (
                <div>
                  <h3 className="text-2xl font-bold text-gray-100 mb-4">Path.net DDoS Protection</h3>
                  <p className="text-gray-60 leading-relaxed mb-6">
                    All traffic is automatically filtered through Path.net's 12Tbps+ scrubbing network. Layer 3/4 and Layer 7
                    attacks are mitigated in real-time without adding latency to legitimate traffic.
                  </p>
                  <div className="flex flex-wrap gap-3">
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">12Tbps+ Capacity</span>
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">L3/L4/L7 Filter</span>
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">Always-On</span>
                  </div>
                </div>
              )}
              {activeTab === 'control' && (
                <div>
                  <h3 className="text-2xl font-bold text-gray-100 mb-4">Virtualizor Control Panel</h3>
                  <p className="text-gray-60 leading-relaxed mb-6">
                    Full VPS management through Virtualizor. Reinstall OS, manage snapshots, monitor resource usage,
                    and access VNC console for emergency recovery. API access available for automation.
                  </p>
                  <div className="flex flex-wrap gap-3">
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">Virtualizor</span>
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">VNC Console</span>
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">REST API</span>
                  </div>
                </div>
              )}
              {activeTab === 'network' && (
                <div>
                  <h3 className="text-2xl font-bold text-gray-100 mb-4">Global Network PoPs</h3>
                  <p className="text-gray-60 leading-relaxed mb-6">
                    Our network spans multiple points of presence with 10Gbps uplinks. Whether you need low-latency
                    gaming servers or global content delivery, our infrastructure delivers.
                  </p>
                  <div className="flex flex-wrap gap-3">
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">10Gbps Uplinks</span>
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">Multiple PoPs</span>
                    <span className="px-3 py-1 bg-gray-5 border border-gray-20 rounded text-sm font-mono text-gray-70">BGP Peering</span>
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>
      </section>

      {/* COMPARISON SECTION */}
      <section className="py-24 bg-gray-10 border-b border-gray-20">
        <div className="bx--grid">
          {/* Section Header */}
          <div className="text-center mb-16 reveal">
            <h2 className="text-3xl md:text-4xl font-light text-gray-100 mb-4">
              Real Performance. <span className="font-semibold">No Hidden Limits.</span>
            </h2>
            <p className="text-gray-60 max-w-2xl mx-auto text-lg font-light leading-relaxed">
              Built for users who need stable RDP and predictable performance — not oversold VPS plans.
              We focus on transparency and real-world usage.
            </p>
          </div>

          {/* Comparison Table */}
          <div className="overflow-x-auto reveal">
            <table className="w-full text-left border-collapse min-w-[900px]">
              <thead>
                <tr>
                  <th className="p-6 w-1/3 align-bottom">
                    <span className="block text-gray-90 font-bold text-lg">Feature Comparison</span>
                    <span className="block text-gray-50 text-xs font-normal mt-2">Based on standard market tiers</span>
                  </th>

                  {/* Competitor Column */}
                  <th className="p-6 w-1/3 align-bottom text-gray-50 font-normal border-b-2 border-gray-20">
                    <div className="mb-2 text-xl">Typical VPS Providers</div>
                    <div className="text-xs text-gray-40">Standard Market Offerings</div>
                  </th>

                  {/* VCHCLOUD Column */}
                  <th className="p-6 w-1/3 align-bottom bg-white relative shadow-2xl border-t-4 border-blue-60 transform -translate-y-2 rounded-t-lg">
                    <div className="flex items-center gap-2 text-xl font-bold text-gray-100">
                      VCH<span className="font-light text-gray-40">CLOUD</span>
                    </div>
                    <div className="text-xs text-blue-60 font-bold mt-1">High Performance Tier</div>
                  </th>
                </tr>
              </thead>
              <tbody className="text-sm">
                {comparisonRows.map((row, index) => (
                  <tr key={index} className="group">
                    <td className={`p-6 ${index === comparisonRows.length - 1 ? 'border-b border-gray-10' : 'border-b border-gray-20'}`}>
                      <strong className="block text-gray-80 text-base">{row.feature}</strong>
                      <span className="text-xs text-gray-50">{row.impact}</span>
                    </td>
                    <td className={`p-6 ${index === comparisonRows.length - 1 ? 'border-b border-gray-10' : 'border-b border-gray-20'} text-gray-60 bg-gray-10/50`}>
                      <div className="flex items-center gap-3">
                        <FontAwesomeIcon icon={faCircle} className="text-gray-40" />
                        <span>{row.competitor}</span>
                      </div>
                    </td>
                    <td className={`p-6 ${index === comparisonRows.length - 1 ? 'border-b border-gray-10 rounded-b-lg shadow-[0_10px_20px_rgba(0,0,0,0.05)]' : 'border-b border-gray-20'} bg-white font-bold text-gray-100 border-x border-gray-10 shadow-[0_0_10px_rgba(0,0,0,0.02)]`}>
                      <div className="flex items-center gap-3">
                        <FontAwesomeIcon icon={faCheck} className="text-green-50" />
                        <span>{row.vchcloud}</span>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>

          {/* Disclaimer & Trust */}
          <div className="mt-12 text-center">
            <p className="text-xs text-gray-50 italic mb-6 max-w-2xl mx-auto">
              * Comparisons are based on publicly available benchmarks and typical market offerings for budget to
              mid-range VPS providers. Actual specifications may vary by provider.
            </p>

            <div className="inline-flex items-center gap-8 py-4 px-8 bg-white border border-gray-20 rounded-full shadow-sm">
              <div className="text-left">
                <span className="block text-2xl font-bold text-gray-100 leading-none">99.9%</span>
                <span className="text-[10px] uppercase font-bold text-gray-50">Uptime Guarantee</span>
              </div>
              <div className="h-8 w-px bg-gray-20"></div>
              <div className="text-left">
                <span className="block text-2xl font-bold text-gray-100 leading-none">NVMe</span>
                <span className="text-[10px] uppercase font-bold text-gray-50">Standard Storage</span>
              </div>
              <div className="h-8 w-px bg-gray-20 hidden sm:block"></div>
              <div className="text-left hidden sm:block">
                <div className="flex text-yellow-40 text-xs mb-1">
                  <FontAwesomeIcon icon={faStar} />
                  <FontAwesomeIcon icon={faStar} />
                  <FontAwesomeIcon icon={faStar} />
                  <FontAwesomeIcon icon={faStar} />
                  <FontAwesomeIcon icon={faStar} />
                </div>
                <span className="text-[10px] uppercase font-bold text-gray-50">User Rated</span>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* KNOWLEDGE HUB & LIVE SUPPORT */}
      <section className="py-24 bg-gray-5 border-b border-gray-20 relative overflow-hidden">
        {/* Background Decor */}
        <div className="absolute top-0 right-0 p-12 opacity-5 pointer-events-none">
          <FontAwesomeIcon icon={faNetworkWired} className="text-9xl" />
        </div>

        <div className="bx--grid relative z-10">
          <div className="grid lg:grid-cols-12 gap-16 items-start">
            {/* LEFT: The Knowledge Engine */}
            <div className="lg:col-span-7 reveal">
              <div className="mb-8">
                <h2 className="text-3xl font-light text-gray-100">Technical Support</h2>
                <p className="text-gray-60 mt-2">Search our engineering docs or open a ticket with the NOC team.</p>
              </div>

              {/* Modern Search Bar */}
              <div className="relative mb-10 group">
                <div className="absolute inset-y-0 left-0 pl-6 flex items-center pointer-events-none">
                  <FontAwesomeIcon icon={faMagnifyingGlass} className="text-gray-40 group-focus-within:text-blue-60 transition-colors" />
                </div>
                <input
                  type="text"
                  placeholder="Search 'DDoS rules', 'IPv6 setup', or 'Billing'..."
                  className="block w-full pl-14 pr-4 py-5 bg-white border border-gray-20 text-gray-100 rounded-xl focus:ring-2 focus:ring-blue-60/20 focus:border-blue-60 outline-none transition-all shadow-sm font-mono text-sm"
                />
                <div className="absolute inset-y-0 right-0 pr-4 flex items-center">
                  <span className="text-xs border border-gray-20 rounded px-2 py-1 text-gray-40 hidden sm:block">CTRL + K</span>
                </div>
              </div>

              {/* Quick Access Grid */}
              <div className="grid sm:grid-cols-2 gap-4 mb-12">
                <Link
                  to="/community"
                  className="group flex items-start p-5 bg-white border border-gray-20 rounded-xl hover:border-blue-60 hover:shadow-lg hover:-translate-y-1 transition-all duration-300"
                >
                  <div className="flex-shrink-0 w-10 h-10 rounded-lg bg-blue-5 text-blue-60 flex items-center justify-center group-hover:bg-blue-60 group-hover:text-white transition-colors">
                    <FontAwesomeIcon icon={faBook} />
                  </div>
                  <div className="ml-4">
                    <h4 className="text-sm font-bold text-gray-100">Documentation</h4>
                    <p className="text-xs text-gray-60 mt-1">Server configuration guides.</p>
                  </div>
                </Link>

                <a
                  href="#"
                  className="group flex items-start p-5 bg-white border border-gray-20 rounded-xl hover:border-purple-60 hover:shadow-lg hover:-translate-y-1 transition-all duration-300"
                >
                  <div className="flex-shrink-0 w-10 h-10 rounded-lg bg-purple-5 text-purple-60 flex items-center justify-center group-hover:bg-purple-60 group-hover:text-white transition-colors">
                    <FontAwesomeIcon icon={faCreditCard} />
                  </div>
                  <div className="ml-4">
                    <h4 className="text-sm font-bold text-gray-100">Billing Portal</h4>
                    <p className="text-xs text-gray-60 mt-1">Invoices & upgrades.</p>
                  </div>
                </a>
              </div>

              {/* Streamlined FAQ */}
              <h3 className="text-sm font-bold text-gray-50 uppercase tracking-widest mb-6">Frequently Asked</h3>
              <div className="space-y-3">
                {/* FAQ Item 1 */}
                <details className="group bg-white border border-gray-20 rounded-lg overflow-hidden">
                  <summary className="flex justify-between items-center p-5 cursor-pointer list-none hover:bg-gray-5 transition">
                    <span className="font-bold text-gray-80 text-sm">Do you offer DDoS Protection?</span>
                    <span className="text-gray-40 transition-transform group-open:rotate-180">
                      <FontAwesomeIcon icon={faChevronDown} />
                    </span>
                  </summary>
                  <div className="px-5 pb-5 text-sm text-gray-60 leading-relaxed border-t border-gray-10 pt-4">
                    Yes. All nodes are protected by our custom <strong>Path.net</strong> filters alongside
                    CosmicGuard. We filter L3/L4 volumetric attacks up to 12Tbps and provide L7 filters for
                    game servers.
                  </div>
                </details>

                {/* FAQ Item 2 */}
                <details className="group bg-white border border-gray-20 rounded-lg overflow-hidden">
                  <summary className="flex justify-between items-center p-5 cursor-pointer list-none hover:bg-gray-5 transition">
                    <span className="font-bold text-gray-80 text-sm">Can I upgrade my VPS later?</span>
                    <span className="text-gray-40 transition-transform group-open:rotate-180">
                      <FontAwesomeIcon icon={faChevronDown} />
                    </span>
                  </summary>
                  <div className="px-5 pb-5 text-sm text-gray-60 leading-relaxed border-t border-gray-10 pt-4">
                    Absolutely. You can scale your CPU, RAM, and Storage instantly via the client area
                    without data loss. A simple reboot is required for CPU changes to take effect.
                  </div>
                </details>

                {/* FAQ Item 3 */}
                <details className="group bg-white border border-gray-20 rounded-lg overflow-hidden">
                  <summary className="flex justify-between items-center p-5 cursor-pointer list-none hover:bg-gray-5 transition">
                    <span className="font-bold text-gray-80 text-sm">What payment methods are accepted?</span>
                    <span className="text-gray-40 transition-transform group-open:rotate-180">
                      <FontAwesomeIcon icon={faChevronDown} />
                    </span>
                  </summary>
                  <div className="px-5 pb-5 text-sm text-gray-60 leading-relaxed border-t border-gray-10 pt-4">
                    We accept Credit Cards (Stripe), PayPal, and various Cryptocurrencies. All payments are
                    processed securely and services are provisioned automatically upon confirmation.
                  </div>
                </details>
              </div>
            </div>

            {/* RIGHT: The "NOC" Dashboard (Fixed) */}
            <div className="lg:col-span-5 reveal" style={{ animationDelay: '0.2s' }}>
              <div className="sticky top-24">
                {/* Dashboard Card */}
                <div className="bg-gray-100 rounded-2xl shadow-2xl overflow-hidden border border-gray-80 text-white relative">
                  {/* Top Highlight Line */}
                  <div className="absolute top-0 w-full h-1 bg-gradient-to-r from-blue-60 via-purple-60 to-blue-60"></div>

                  {/* Card Header */}
                  <div className="px-6 py-5 border-b border-gray-80 flex justify-between items-center bg-[#1a1a1a]">
                    <div className="flex items-center gap-3">
                      <div className="relative">
                        <div className="w-3 h-3 bg-green-50 rounded-full animate-pulse"></div>
                        <div className="absolute inset-0 w-3 h-3 bg-green-50 rounded-full animate-ping opacity-75"></div>
                      </div>
                      <h3 className="font-mono font-bold text-sm tracking-wide">SYSTEM_STATUS</h3>
                    </div>
                    <div className="flex items-center gap-2 text-[10px] font-mono text-gray-50 bg-gray-80 px-2 py-1 rounded">
                      <FontAwesomeIcon icon={faClock} /> UTC 15:13
                    </div>
                  </div>

                  {/* Dashboard Body */}
                  <div className="p-6 bg-[#111]">
                    {/* Load Graphic */}
                    <div className="mb-8">
                      <div className="flex justify-between text-xs font-mono text-gray-50 mb-2">
                        <span>NETWORK_LOAD</span>
                        <span className="text-blue-40">34%</span>
                      </div>
                      <div className="flex gap-1 h-8 items-end">
                        <div className="w-1 bg-blue-60/20 h-[40%]"></div>
                        <div className="w-1 bg-blue-60/40 h-[60%]"></div>
                        <div className="w-1 bg-blue-60/30 h-[30%]"></div>
                        <div className="w-1 bg-blue-60/60 h-[70%]"></div>
                        <div className="w-1 bg-blue-60 h-[50%]"></div>
                        <div className="w-1 bg-blue-60 h-[80%]"></div>
                        <div className="w-1 bg-blue-60 h-[45%]"></div>
                        <div className="w-1 bg-blue-60/50 h-[60%]"></div>
                        <div className="w-1 bg-blue-60/30 h-[30%]"></div>
                        <div className="w-1 bg-blue-60/20 h-[20%]"></div>
                        <div className="flex-grow border-b border-gray-80 relative -top-[1px]"></div>
                      </div>
                    </div>

                    {/* Server Status List */}
                    <div className="space-y-4 mb-8">
                      {/* Mumbai */}
                      <div className="flex justify-between items-center group">
                        <div className="flex items-center gap-3">
                          <img src="https://flagcdn.com/w20/in.png" alt="India" className="w-5 rounded-sm opacity-80" />
                          <div>
                            <p className="text-sm font-bold text-gray-20 group-hover:text-white transition">Mumbai, IN</p>
                            <p className="text-[10px] text-gray-50 font-mono">ap-south-1</p>
                          </div>
                        </div>
                        <div className="text-right">
                          <div className="px-2 py-0.5 rounded bg-green-900/30 border border-green-900 text-green-50 text-[10px] font-bold uppercase inline-block mb-1">
                            Operational
                          </div>
                          <p className="text-[10px] text-gray-60 font-mono">4ms Latency</p>
                        </div>
                      </div>

                      {/* Frankfurt */}
                      <div className="flex justify-between items-center group opacity-60">
                        <div className="flex items-center gap-3">
                          <img src="https://flagcdn.com/w20/de.png" alt="Germany" className="w-5 rounded-sm opacity-80" />
                          <div>
                            <p className="text-sm font-bold text-gray-20">Frankfurt, DE</p>
                            <p className="text-[10px] text-gray-50 font-mono">eu-central-1</p>
                          </div>
                        </div>
                        <div className="text-right">
                          <div className="px-2 py-0.5 rounded bg-yellow-900/30 border border-yellow-900 text-yellow-50 text-[10px] font-bold uppercase inline-block mb-1">
                            Provisioning
                          </div>
                          <p className="text-[10px] text-gray-60 font-mono">--ms Latency</p>
                        </div>
                      </div>
                    </div>

                    {/* Support Performance Box */}
                    <div className="bg-gray-90 rounded-xl p-4 border border-gray-80 flex items-center justify-between mb-6">
                      <div>
                        <p className="text-[10px] text-gray-50 uppercase tracking-wide mb-1">Avg. Reply Time</p>
                        <p className="text-xl font-bold text-white">&lt; 2 Minutes</p>
                      </div>

                      {/* Clean Live Status Indicator */}
                      <div className="flex items-center gap-3 px-3 py-2 bg-gray-80 rounded-lg border border-gray-70">
                        <div className="relative flex h-2 w-2">
                          <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-green-40 opacity-75"></span>
                          <span className="relative inline-flex rounded-full h-2 w-2 bg-green-50"></span>
                        </div>
                        <span className="text-xs font-bold text-gray-30">Agents Online</span>
                      </div>
                    </div>

                    {/* Action Button */}
                    <Link
                      to="/support/ticket"
                      className="block w-full py-4 bg-white hover:bg-gray-10 text-gray-100 font-bold text-sm text-center rounded-lg transition-all duration-300 shadow-[0_0_20px_rgba(255,255,255,0.1)] hover:shadow-[0_0_25px_rgba(255,255,255,0.2)]"
                    >
                      <FontAwesomeIcon icon={faTicket} className="mr-2 text-blue-60" /> Open Priority Ticket
                    </Link>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

    </div>
  );
};

export default Home;
