import { useState } from 'react';
import { Link } from 'react-router-dom';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import {
  faMagnifyingGlass,
  faBook,
  faServer,
  faShieldHalved,
  faReceipt,
  faArrowRight,
  faLayerGroup
} from '@fortawesome/free-solid-svg-icons';
import Navbar from '../components/Navbar';
import Footer from '../components/Footer';

const Docs = () => {
  const [searchQuery, setSearchQuery] = useState('');

  const categories = [
    {
      icon: faBook,
      title: 'Getting Started',
      description: 'New to HostCraft? Start here to learn the basics.',
      color: '#0f62fe',
      links: [
        { title: 'First RDP Login Guide', href: '/community' },
        { title: 'Service Level Agreement', href: '/community' },
        { title: 'Account Setup', href: '/community' }
      ]
    },
    {
      icon: faServer,
      title: 'VPS Management',
      description: 'Learn how to manage your virtual private server.',
      color: '#8a3ffc',
      links: [
        { title: 'Reset Root Password', href: '/community' },
        { title: 'VNC Console Access', href: '/community' },
        { title: 'OS Reinstallation', href: '/community' }
      ]
    },
    {
      icon: faShieldHalved,
      title: 'Security',
      description: 'Best practices for securing your infrastructure.',
      color: '#24a148',
      links: [
        { title: 'Path.net DDoS Filters', href: '/community' },
        { title: 'Windows Firewall Basics', href: '/community' },
        { title: 'Configure UFW (Linux)', href: '/community' }
      ]
    },
    {
      icon: faReceipt,
      title: 'Billing',
      description: 'Payment methods, invoices, and upgrades.',
      color: '#ba4e00',
      links: [
        { title: 'Upgrade Resources', href: '/community' },
        { title: 'Payment Methods', href: '/community' },
        { title: 'Refund Policy', href: '/legal/refund' }
      ]
    }
  ];

  const handleSearch = (e) => {
    e.preventDefault();
    if (searchQuery.trim()) {
      window.location.href = `/community?q=${encodeURIComponent(searchQuery)}`;
    }
  };

  return (
    <div className="pt-14 bg-[#f4f4f4] min-h-screen">
      {/* Hero Section */}
      <section className="bg-[#161616] py-20 border-b border-[#393939] relative overflow-hidden">
        {/* Technical Grid Pattern */}
        <div
          className="absolute inset-0 z-0 opacity-10 pointer-events-none"
          style={{
            backgroundImage: 'linear-gradient(#ffffff 1px, transparent 1px), linear-gradient(to right, #ffffff 1px, transparent 1px)',
            backgroundSize: '40px 40px'
          }}
        ></div>

        <div className="bx--grid relative z-10">
          <div className="max-w-3xl mx-auto text-center">
            <div className="inline-flex items-center gap-2 px-3 py-1 bg-[#0f62fe]/10 border border-[#0f62fe] text-[#0f62fe] text-xs font-bold uppercase tracking-wider mb-6 rounded-full">
              <FontAwesomeIcon icon={faBook} />
              Documentation
            </div>

            <h1 className="text-4xl md:text-5xl font-light text-white mb-6 tracking-tight">
              HostCraft <span className="font-bold text-[#0f62fe]">Docs</span>
            </h1>

            <p className="text-lg text-[#a8a8a8] mb-8">
              Find guides, tutorials, and technical documentation for managing your infrastructure.
            </p>

            {/* Search Input */}
            <form onSubmit={handleSearch} className="relative max-w-xl mx-auto">
              <div className="absolute inset-y-0 left-0 pl-5 flex items-center pointer-events-none">
                <FontAwesomeIcon icon={faMagnifyingGlass} className="text-[#6f6f6f]" />
              </div>
              <input
                type="text"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                placeholder="Search documentation..."
                className="w-full pl-12 pr-4 py-4 bg-[#262626] border border-[#393939] text-white rounded-lg focus:border-[#0f62fe] focus:ring-1 focus:ring-[#0f62fe] outline-none transition placeholder-[#6f6f6f]"
              />
            </form>
          </div>
        </div>
      </section>

      {/* Categories Grid */}
      <section className="py-20">
        <div className="bx--grid">
          <div className="grid md:grid-cols-2 gap-8">
            {categories.map((category, index) => (
              <div
                key={index}
                className="bg-white border border-[#e0e0e0] rounded-xl p-8 hover:shadow-lg transition"
              >
                <div
                  className="w-14 h-14 rounded-lg flex items-center justify-center mb-6"
                  style={{ backgroundColor: `${category.color}10`, color: category.color }}
                >
                  <FontAwesomeIcon icon={category.icon} className="text-2xl" />
                </div>

                <h2 className="text-xl font-bold text-[#161616] mb-2">{category.title}</h2>
                <p className="text-[#6f6f6f] mb-6">{category.description}</p>

                <ul className="space-y-3">
                  {category.links.map((link, idx) => (
                    <li key={idx}>
                      <Link
                        to={link.href}
                        className="flex items-center gap-2 text-[#525252] hover:text-[#0f62fe] transition group"
                      >
                        <FontAwesomeIcon
                          icon={faArrowRight}
                          className="text-xs text-[#a8a8a8] group-hover:text-[#0f62fe] group-hover:translate-x-1 transition-all"
                        />
                        {link.title}
                      </Link>
                    </li>
                  ))}
                </ul>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Community Guides CTA */}
      <section className="py-16 bg-white border-y border-[#e0e0e0]">
        <div className="bx--grid">
          <div className="flex flex-col md:flex-row items-center justify-between gap-8">
            <div>
              <div className="flex items-center gap-3 mb-3">
                <FontAwesomeIcon icon={faLayerGroup} className="text-[#8a3ffc] text-xl" />
                <h2 className="text-xl font-bold text-[#161616]">Community Guides</h2>
              </div>
              <p className="text-[#6f6f6f] max-w-xl">
                Browse community-contributed tutorials and guides. Learn from real-world examples
                and best practices shared by other users.
              </p>
            </div>
            <Link
              to="/community"
              className="px-8 py-4 bg-[#8a3ffc] text-white font-bold rounded hover:bg-[#6929c4] transition whitespace-nowrap"
            >
              Browse Guides
            </Link>
          </div>
        </div>
      </section>

      {/* Quick Links */}
      <section className="py-20">
        <div className="bx--grid">
          <div className="max-w-4xl mx-auto">
            <h2 className="text-2xl font-bold text-[#161616] mb-8 text-center">Quick Links</h2>

            <div className="grid sm:grid-cols-2 lg:grid-cols-3 gap-4">
              {[
                { title: 'API Reference', href: '#' },
                { title: 'Status Page', href: '#' },
                { title: 'Network Test Files', href: '#' },
                { title: 'Terms of Service', href: '/legal/terms' },
                { title: 'Privacy Policy', href: '/legal/privacy' },
                { title: 'Acceptable Use', href: '/legal/aup' }
              ].map((link, index) => (
                <Link
                  key={index}
                  to={link.href}
                  className="flex items-center justify-between p-4 bg-white border border-[#e0e0e0] rounded-lg hover:border-[#0f62fe] hover:shadow-md transition group"
                >
                  <span className="font-medium text-[#161616]">{link.title}</span>
                  <FontAwesomeIcon
                    icon={faArrowRight}
                    className="text-[#a8a8a8] group-hover:text-[#0f62fe] group-hover:translate-x-1 transition-all"
                  />
                </Link>
              ))}
            </div>
          </div>
        </div>
      </section>
    </div>
  );
};

export default Docs;
