import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import {
  faServer,
  faFileLines,
  faPlus,
  faRightFromBracket,
  faGauge,
  faReceipt,
  faHeadset,
  faLayerGroup
} from '@fortawesome/free-solid-svg-icons';

const Dashboard = () => {
  const [loading, setLoading] = useState(true);
  const [userData, setUserData] = useState(null);
  const [stats, setStats] = useState({
    activeInstances: 0,
    unpaidInvoices: 0
  });

  useEffect(() => {
    // Simulate API call
    setTimeout(() => {
      setUserData({
        name: 'Demo User',
        role: 'Client'
      });
      setStats({
        activeInstances: 0,
        unpaidInvoices: 0
      });
      setLoading(false);
    }, 1000);
  }, []);

  const sidebarLinks = [
    { icon: faGauge, label: 'Overview', active: true },
    { icon: faLayerGroup, label: 'Instances', active: false },
    { icon: faReceipt, label: 'Billing', active: false },
    { icon: faHeadset, label: 'Support', active: false }
  ];

  return (
    <div className="min-h-screen bg-[#f4f4f4] flex">
      {/* Sidebar */}
      <aside className="hidden md:flex flex-col w-64 bg-[#161616] border-r border-[#393939]">
        {/* Brand */}
        <div className="p-6 border-b border-[#393939]">
          <Link to="/" className="flex items-center gap-3">
            <span className="font-bold text-lg text-white">
              VCH<span className="font-light text-[#a8a8a8]">CLOUD</span>
            </span>
          </Link>
        </div>

        {/* Navigation */}
        <nav className="flex-1 p-4">
          <ul className="space-y-2">
            {sidebarLinks.map((link, index) => (
              <li key={index}>
                <a
                  href="#"
                  className={`flex items-center gap-3 px-4 py-3 rounded-lg transition ${
                    link.active
                      ? 'bg-[#0f62fe]/10 text-[#0f62fe] border-l-4 border-[#0f62fe]'
                      : 'text-[#a8a8a8] hover:bg-[#262626] hover:text-white'
                  }`}
                >
                  <FontAwesomeIcon icon={link.icon} className="w-5" />
                  <span className="font-medium">{link.label}</span>
                </a>
              </li>
            ))}
          </ul>
        </nav>

        {/* Logout */}
        <div className="p-4 border-t border-[#393939]">
          <Link
            to="/"
            className="flex items-center gap-3 px-4 py-3 text-[#da1e28] hover:bg-[#262626] rounded-lg transition"
          >
            <FontAwesomeIcon icon={faRightFromBracket} className="w-5" />
            <span className="font-medium">Logout</span>
          </Link>
        </div>
      </aside>

      {/* Main Content */}
      <main className="flex-1">
        {/* Header */}
        <header className="bg-white border-b border-[#e0e0e0] px-8 py-6">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold text-[#161616]">Dashboard</h1>
              <p className="text-[#6f6f6f] text-sm">Manage your cloud infrastructure</p>
            </div>

            {/* User Profile */}
            {loading ? (
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 rounded-full bg-[#e0e0e0] animate-pulse"></div>
                <div>
                  <div className="w-24 h-4 bg-[#e0e0e0] rounded animate-pulse mb-1"></div>
                  <div className="w-16 h-3 bg-[#e0e0e0] rounded animate-pulse"></div>
                </div>
              </div>
            ) : (
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 rounded-full bg-[#0f62fe] flex items-center justify-center text-white font-bold">
                  {userData?.name?.charAt(0) || 'U'}
                </div>
                <div>
                  <div className="font-medium text-[#161616]">{userData?.name}</div>
                  <div className="text-xs text-[#6f6f6f]">{userData?.role}</div>
                </div>
              </div>
            )}
          </div>
        </header>

        {/* Content */}
        <div className="p-8">
          {/* Stats Cards */}
          {loading ? (
            <div className="grid md:grid-cols-3 gap-6 mb-8">
              {[1, 2, 3].map((i) => (
                <div key={i} className="bg-white border border-[#e0e0e0] rounded-lg p-6 animate-pulse">
                  <div className="w-12 h-12 bg-[#e0e0e0] rounded mb-4"></div>
                  <div className="w-20 h-6 bg-[#e0e0e0] rounded mb-2"></div>
                  <div className="w-32 h-4 bg-[#e0e0e0] rounded"></div>
                </div>
              ))}
            </div>
          ) : (
            <div className="grid md:grid-cols-3 gap-6 mb-8">
              {/* Active Instances */}
              <div className="bg-white border border-[#e0e0e0] rounded-lg p-6">
                <div className="w-12 h-12 bg-[#0f62fe]/10 text-[#0f62fe] rounded-lg flex items-center justify-center mb-4">
                  <FontAwesomeIcon icon={faServer} className="text-xl" />
                </div>
                <div className="text-3xl font-bold text-[#161616] mb-1">{stats.activeInstances}</div>
                <div className="text-[#6f6f6f] text-sm">Active Instances</div>
              </div>

              {/* Unpaid Invoices */}
              <div className="bg-white border border-[#e0e0e0] rounded-lg p-6">
                <div className="w-12 h-12 bg-[#8a3ffc]/10 text-[#8a3ffc] rounded-lg flex items-center justify-center mb-4">
                  <FontAwesomeIcon icon={faFileLines} className="text-xl" />
                </div>
                <div className="text-3xl font-bold text-[#161616] mb-1">{stats.unpaidInvoices}</div>
                <div className="text-[#6f6f6f] text-sm">Unpaid Invoices</div>
              </div>

              {/* System Status */}
              <div className="bg-white border border-[#e0e0e0] rounded-lg p-6">
                <div className="w-12 h-12 bg-[#24a148]/10 text-[#24a148] rounded-lg flex items-center justify-center mb-4">
                  <span className="relative flex h-3 w-3">
                    <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-[#24a148] opacity-75"></span>
                    <span className="relative inline-flex rounded-full h-3 w-3 bg-[#24a148]"></span>
                  </span>
                </div>
                <div className="text-xl font-bold text-[#24a148] mb-1">Operational</div>
                <div className="text-[#6f6f6f] text-sm">All Systems Online</div>
              </div>
            </div>
          )}

          {/* Empty State */}
          {!loading && stats.activeInstances === 0 && (
            <div className="bg-white border border-[#e0e0e0] rounded-lg p-12 text-center">
              <div className="w-20 h-20 bg-[#f4f4f4] rounded-full flex items-center justify-center mx-auto mb-6">
                <FontAwesomeIcon icon={faPlus} className="text-3xl text-[#6f6f6f]" />
              </div>
              <h3 className="text-xl font-bold text-[#161616] mb-2">No Active Services</h3>
              <p className="text-[#6f6f6f] mb-6 max-w-md mx-auto">
                You don't have any active VPS instances. Deploy your first instance to get started with VCHCLOUD.
              </p>
              <Link
                to="/pricing/vps"
                className="inline-flex items-center gap-2 px-6 py-3 bg-[#0f62fe] text-white font-bold rounded hover:bg-[#0043ce] transition"
              >
                <FontAwesomeIcon icon={faPlus} />
                Deploy New Instance
              </Link>
            </div>
          )}
        </div>
      </main>
    </div>
  );
};

export default Dashboard;
