import { useState, useEffect, useRef } from 'react';

const Terminal = () => {
  const [lines, setLines] = useState([]);
  const terminalRef = useRef(null);

  const commands = [
    {
      command: 'admin@hostcraft:~$ ./start_server.sh -game minecraft',
      output: [
        '[OK] Server instance created.',
        '[OK] Loading world chunks...',
        '[OK] Minecraft 1.20 ready on :25565'
      ]
    },
    {
      command: 'admin@hostcraft:~$ npm start discord-bot',
      output: [
        '> bot@1.0.0 start',
        '> node index.js',
        '[READY] Bot online as HostBot#1234'
      ]
    },
    {
      command: 'admin@hostcraft:~$ ping -c 3 sg.hostcraft.net',
      output: [
        'PING sg.hostcraft.net (103.42.x.x): 56 data bytes',
        '64 bytes: icmp_seq=0 ttl=55 time=2.4ms',
        '64 bytes: icmp_seq=1 ttl=55 time=2.1ms',
        '64 bytes: icmp_seq=2 ttl=55 time=2.3ms'
      ]
    }
  ];

  useEffect(() => {
    let currentCommandIndex = 0;
    let currentCharIndex = 0;
    let showingOutput = false;
    let outputLineIndex = 0;
    let isTyping = true;

    const typeWriter = () => {
      if (!isTyping) return;

      const currentCmd = commands[currentCommandIndex];

      if (!showingOutput) {
        // Typing the command
        if (currentCharIndex < currentCmd.command.length) {
          setLines(prev => {
            const newLines = [...prev];
            if (newLines.length === 0 || newLines[newLines.length - 1].type !== 'typing') {
              newLines.push({ type: 'typing', text: currentCmd.command.substring(0, currentCharIndex + 1) });
            } else {
              newLines[newLines.length - 1] = { type: 'typing', text: currentCmd.command.substring(0, currentCharIndex + 1) };
            }
            return newLines;
          });
          currentCharIndex++;
          setTimeout(typeWriter, 30);
        } else {
          // Command finished typing, show output
          setLines(prev => {
            const newLines = [...prev];
            newLines[newLines.length - 1] = { type: 'command', text: currentCmd.command };
            return newLines;
          });
          showingOutput = true;
          outputLineIndex = 0;
          setTimeout(typeWriter, 200);
        }
      } else {
        // Showing output lines
        if (outputLineIndex < currentCmd.output.length) {
          setLines(prev => [...prev, { type: 'output', text: currentCmd.output[outputLineIndex] }]);
          outputLineIndex++;
          setTimeout(typeWriter, 100);
        } else {
          // Move to next command
          currentCommandIndex = (currentCommandIndex + 1) % commands.length;
          currentCharIndex = 0;
          showingOutput = false;

          if (currentCommandIndex === 0) {
            // Reset terminal after all commands
            setTimeout(() => {
              setLines([]);
              setTimeout(typeWriter, 500);
            }, 2000);
          } else {
            setTimeout(typeWriter, 1000);
          }
        }
      }
    };

    // Start typing
    const timer = setTimeout(typeWriter, 500);

    // Cleanup
    return () => {
      isTyping = false;
      clearTimeout(timer);
    };
  }, []);

  // Auto-scroll to bottom
  useEffect(() => {
    if (terminalRef.current) {
      terminalRef.current.scrollTop = terminalRef.current.scrollHeight;
    }
  }, [lines]);

  return (
    <div className="bg-gray-100 border border-gray-80 rounded-lg overflow-hidden shadow-2xl">
      {/* Terminal Header */}
      <div className="bg-gray-90 px-4 py-2 flex items-center gap-2 border-b border-gray-80">
        <div className="flex gap-1.5">
          <div className="w-3 h-3 rounded-full bg-red-50"></div>
          <div className="w-3 h-3 rounded-full bg-yellow-40"></div>
          <div className="w-3 h-3 rounded-full bg-green-40"></div>
        </div>
        <span className="text-[10px] font-mono text-gray-60 ml-2">admin@hostcraft ~ bash</span>
      </div>

      {/* Terminal Content */}
      <div
        ref={terminalRef}
        className="p-4 h-64 overflow-y-auto font-mono text-sm"
        id="terminal-content"
      >
        {lines.map((line, index) => (
          <div key={index} className="mb-1">
            {line.type === 'typing' ? (
              <span className="text-green-40">
                {line.text}
                <span className="cursor"></span>
              </span>
            ) : line.type === 'command' ? (
              <span className="text-green-40">{line.text}</span>
            ) : (
              <span className="text-gray-40">{line.text}</span>
            )}
          </div>
        ))}
        {lines.length === 0 && (
          <span className="text-green-40">
            <span className="cursor"></span>
          </span>
        )}
      </div>
    </div>
  );
};

export default Terminal;
