import { useState } from 'react';
import { Link } from 'react-router-dom';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import {
  faBars,
  faXmark,
  faChevronDown,
  faServer,
  faCloud,
  faArrowRight,
  faSun,
  faMoon
} from '@fortawesome/free-solid-svg-icons';
import { faUser } from '@fortawesome/free-regular-svg-icons';
import { useTheme } from '../context/ThemeContext';

const Navbar = () => {
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const { isDark, toggleTheme } = useTheme();

  const toggleMobileMenu = () => {
    setMobileMenuOpen(!mobileMenuOpen);
    if (!mobileMenuOpen) {
      document.body.style.overflow = 'hidden';
    } else {
      document.body.style.overflow = '';
    }
  };

  const closeMobileMenu = () => {
    setMobileMenuOpen(false);
    document.body.style.overflow = '';
  };

  return (
    <>
      {/* NAVIGATION */}
      <nav
        className="fixed w-full top-0 z-50 border-b border-[#393939] bg-[#161616]/95 backdrop-blur-md h-14 transition-all duration-300"
        id="navbar"
      >
        <div className="bx--grid h-full">
          <div className="flex items-center justify-between h-full">
            {/* Left: Brand & Desktop Links */}
            <div className="flex items-center gap-8 h-full">
              {/* Brand Logo */}
              <Link
                to="/"
                className="flex items-center gap-3 h-full mr-4 group"
              >
                <span className="font-sans font-bold text-lg tracking-tight text-white">
                  VCH<span className="font-light text-[#a8a8a8]">CLOUD</span>
                </span>
              </Link>

              {/* Desktop Menu */}
              <div className="hidden md:flex items-center h-full space-x-1">
                <Link
                  to="/pricing/vps"
                  className="h-full flex items-center px-4 text-sm font-medium text-[#c6c6c6] hover:text-white hover:bg-white/5 transition-colors border-b-2 border-transparent hover:border-[#0f62fe]"
                >
                  Pricing
                </Link>
                <a
                  href="/#infrastructure"
                  className="h-full flex items-center px-4 text-sm font-medium text-[#c6c6c6] hover:text-white hover:bg-white/5 transition-colors border-b-2 border-transparent hover:border-[#0f62fe]"
                >
                  Infrastructure
                </a>
                <a
                  href="/#features"
                  className="h-full flex items-center px-4 text-sm font-medium text-[#c6c6c6] hover:text-white hover:bg-white/5 transition-colors border-b-2 border-transparent hover:border-[#0f62fe]"
                >
                  Solutions
                </a>
                <Link
                  to="/docs"
                  className="h-full flex items-center px-4 text-sm font-medium text-[#c6c6c6] hover:text-white hover:bg-white/5 transition-colors border-b-2 border-transparent hover:border-[#0f62fe]"
                >
                  Docs
                </Link>
                <Link
                  to="/support"
                  className="h-full flex items-center px-4 text-sm font-medium text-[#c6c6c6] hover:text-white hover:bg-white/5 transition-colors border-b-2 border-transparent hover:border-[#0f62fe]"
                >
                  Support
                </Link>
              </div>
            </div>

            {/* Right: Utilities & CTA */}
            <div className="flex items-center h-full gap-4">
              {/* Theme Toggle */}
              <button
                onClick={toggleTheme}
                className="w-9 h-9 flex items-center justify-center text-[#c6c6c6] hover:text-white hover:bg-[#393939] rounded transition-all duration-300"
                aria-label={isDark ? 'Switch to light mode' : 'Switch to dark mode'}
                title={isDark ? 'Switch to light mode' : 'Switch to dark mode'}
              >
                <FontAwesomeIcon
                  icon={isDark ? faSun : faMoon}
                  className={`text-lg transition-transform duration-300 ${isDark ? 'rotate-0' : 'rotate-0'}`}
                />
              </button>

              {/* System Status (Desktop) */}
              <div className="hidden lg:flex items-center gap-2 px-3 py-1 bg-[#262626] rounded border border-[#393939]">
                <span className="relative flex h-2 w-2">
                  <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-[#42be65] opacity-75"></span>
                  <span className="relative inline-flex rounded-full h-2 w-2 bg-[#24a148]"></span>
                </span>
                <span className="text-[10px] font-mono text-[#a8a8a8]">SYSTEMS: OPERATIONAL</span>
              </div>

              <div className="w-px h-6 bg-[#393939] hidden md:block"></div>

              <Link
                to="/login"
                className="hidden md:flex items-center gap-2 text-sm text-[#c6c6c6] hover:text-white transition font-medium"
              >
                <FontAwesomeIcon icon={faUser} /> Login
              </Link>

              {/* Client Area Dropdown */}
              <div className="hidden md:block relative group">
                <button className="flex items-center gap-2 px-5 py-2 bg-white text-[#161616] text-xs font-bold uppercase tracking-wider hover:bg-[#0f62fe] hover:text-white transition shadow-lg hover:shadow-[#0f62fe]/20 rounded-sm">
                  <span>Client Area</span>
                  <FontAwesomeIcon
                    icon={faChevronDown}
                    className="text-[10px] transition-transform group-hover:rotate-180"
                  />
                </button>

                {/* Dropdown Menu */}
                <div className="absolute right-0 top-full pt-2 w-48 opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200 transform group-hover:translate-y-0 translate-y-2 z-50">
                  <div className="bg-[#161616] border border-[#393939] rounded shadow-xl overflow-hidden">
                    <Link
                      to="/pricing/vps"
                      className="block px-4 py-3 text-sm text-[#c6c6c6] hover:bg-white/5 hover:text-white transition border-b border-[#393939]"
                    >
                      <FontAwesomeIcon icon={faServer} className="w-5 text-[#0f62fe] mr-2" /> VPS Hosting
                    </Link>
                    <a
                      href="#"
                      className="block px-4 py-3 text-sm text-[#c6c6c6] hover:bg-white/5 hover:text-white transition"
                    >
                      <FontAwesomeIcon icon={faCloud} className="w-5 text-[#8a3ffc] mr-2" /> Web Hosting
                    </a>
                  </div>
                </div>
              </div>

              {/* Mobile Menu Button */}
              <button
                onClick={toggleMobileMenu}
                type="button"
                className="md:hidden w-10 h-10 flex items-center justify-center text-[#c6c6c6] hover:text-white hover:bg-[#393939] rounded transition"
              >
                <FontAwesomeIcon icon={mobileMenuOpen ? faXmark : faBars} className="text-xl" />
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* MOBILE DRAWER MENU */}
      <div
        className={`fixed inset-0 bg-[#0f0f0f]/95 backdrop-blur-xl z-40 pt-20 transform transition-transform duration-500 md:hidden flex flex-col border-l border-[#393939] ${
          mobileMenuOpen ? 'translate-x-0' : 'translate-x-full'
        }`}
      >
        {/* Technical Background */}
        <div
          className="absolute inset-0 z-0 opacity-[0.03] pointer-events-none"
          style={{
            backgroundImage: 'linear-gradient(#444 1px, transparent 1px), linear-gradient(to right, #444 1px, transparent 1px)',
            backgroundSize: '30px 30px'
          }}
        ></div>

        {/* Gradient Glow */}
        <div className="absolute top-0 right-0 w-full h-[500px] bg-[#0f62fe]/10 blur-[120px] pointer-events-none"></div>

        <div className="relative z-10 flex-1 px-8 flex flex-col">
          {/* Mobile Links */}
          <div className="space-y-3 mb-10">
            <div className="text-[10px] font-bold text-[#8d8d8d] uppercase tracking-widest mb-6 px-1">
              Navigation
            </div>

            <Link
              to="/pricing/vps"
              onClick={closeMobileMenu}
              className="flex items-center justify-between p-5 border border-[#393939] bg-[#262626]/50 hover:bg-[#393939] hover:border-[#6f6f6f] text-[#c6c6c6] hover:text-white rounded-xl group transition-all duration-300"
            >
              <span className="font-bold text-lg tracking-tight">Pricing</span>
              <FontAwesomeIcon
                icon={faArrowRight}
                className="text-[#6f6f6f] group-hover:text-white transform group-hover:translate-x-1 transition-all"
              />
            </Link>

            <Link
              to="/docs"
              onClick={closeMobileMenu}
              className="flex items-center justify-between p-5 border border-[#393939] bg-[#262626]/50 hover:bg-[#393939] hover:border-[#6f6f6f] text-[#c6c6c6] hover:text-white rounded-xl group transition-all duration-300"
            >
              <span className="font-bold text-lg tracking-tight">Documentation</span>
              <FontAwesomeIcon
                icon={faArrowRight}
                className="text-[#6f6f6f] group-hover:text-white transform group-hover:translate-x-1 transition-all"
              />
            </Link>

            <Link
              to="/support"
              onClick={closeMobileMenu}
              className="flex items-center justify-between p-5 border border-[#393939] bg-[#262626]/50 hover:bg-[#393939] hover:border-[#6f6f6f] text-[#c6c6c6] hover:text-white rounded-xl group transition-all duration-300"
            >
              <span className="font-bold text-lg tracking-tight">Support Center</span>
              <FontAwesomeIcon
                icon={faArrowRight}
                className="text-[#6f6f6f] group-hover:text-white transform group-hover:translate-x-1 transition-all"
              />
            </Link>
          </div>

          {/* Mobile Utilities */}
          <div className="grid grid-cols-2 gap-4 mt-auto mb-10">
            <Link
              to="/login"
              onClick={closeMobileMenu}
              className="flex items-center justify-center py-4 border border-[#525252] text-[#c6c6c6] hover:text-white hover:border-white font-bold rounded-lg transition-all text-sm uppercase tracking-wide"
            >
              Login
            </Link>
            <Link
              to="/pricing/vps"
              onClick={closeMobileMenu}
              className="flex items-center justify-center py-4 bg-white text-black font-bold rounded-lg hover:bg-[#e0e0e0] transition-all text-sm uppercase tracking-wide shadow-lg shadow-white/10"
            >
              Client Area
            </Link>
          </div>

          {/* Mobile Status */}
          <div className="py-6 border-t border-[#393939] text-center">
            <div className="inline-flex items-center gap-2 px-3 py-1.5 bg-[#022d0d]/30 border border-[#044317] rounded-full">
              <span className="relative flex h-2 w-2">
                <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-[#24a148] opacity-75"></span>
                <span className="relative inline-flex rounded-full h-2 w-2 bg-[#24a148]"></span>
              </span>
              <span className="text-[10px] font-bold text-[#42be65] tracking-wider">SYSTEMS OPERATIONAL</span>
            </div>
          </div>
        </div>
      </div>
    </>
  );
};

export default Navbar;
